# Copyright 2024, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

import calendar
import datetime
import os
import time

from logger.utils.timezone import get_time_zone


class TestTimezone:
    def test_get_timezone_with_japan(self):
        os.environ["TZ"] = "Asia/Tokyo"
        time.tzset()
        actual = get_time_zone()
        assert actual == "UTC+0900"

    @staticmethod
    def _get_last_sunday(year: int, month: int) -> int:
        last = calendar.monthrange(year, month)[1]
        last7 = range(last - 6, last + 1)
        wday0 = calendar.weekday(year, month, last7[0])
        wdays = [w % 7 for w in range(wday0, wday0 + 7)]
        return last7[wdays.index(6)]  # 6:Sunday

    def test_get_last_sunday(self):
        # 2024/10 の最終日曜日は 27 日
        # The last Sunday of October 2024 is the 27th
        assert TestTimezone._get_last_sunday(2024, 10) == 27

    def test_get_timezone_with_berlin(self):
        os.environ["TZ"] = "Europe/Berlin"
        time.tzset()
        actual = get_time_zone()

        # ベルリンでは「3月最終日曜日2:00〜10月最終日曜日3:00」がサマータイム
        # In Berlin, daylight saving time is from the last Sunday in March at 2:00
        # to the last Sunday in October at 3:00
        now = datetime.datetime.now()
        ds_beg = datetime.datetime(
            year=now.year,
            month=3,
            day=TestTimezone._get_last_sunday(now.year, 3),
            hour=2,
        )
        ds_end = datetime.datetime(
            year=now.year,
            month=10,
            day=TestTimezone._get_last_sunday(now.year, 10),
            hour=3,
        )

        # ここの大小関係は厳密には違うかも..
        # The comparison here might not be exact..
        if ds_beg <= now and now < ds_end:
            assert actual == "UTC+0200"
        else:
            assert actual == "UTC+0100"

    def test_get_timezone_with_columbia(self):
        os.environ["TZ"] = "America/Bogota"
        time.tzset()
        actual = get_time_zone()
        assert actual == "UTC-0500"
