# Copyright 2024, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

import os

import pytest

from logger.utils.validator import Validator


class TestValidator:
    class TestOneOf:
        def test_none_should_return_false(self):
            assert not Validator().one_of(None, "TEST_KEY", [])

        def test_item_in_list_should_return_true(self):
            assert Validator().one_of("hoge", "TEST_KEY", ["hoge", "fuga"])

        def test_item_not_in_list_should_return_false(self):
            assert not Validator().one_of("hoge", "TEST_KEY", ["fuga", "piyo"])

        def test_int_item_list_should_not_raise_error(self):
            # intのリストをjoinするとエラーになるため、修正されたことを確認する
            # Confirm that joining a list of ints no longer raises an error
            try:
                Validator().one_of(1, "TEST_KEY", [2, 3])
            except Exception:
                pytest.fail("Validator one_of raise unexpected Error")

    class TestBetween:
        def test_none_should_return_false(self):
            assert not Validator().between(None, "TEST_KEY", 1, 100)

        def test_item_in_range_should_return_true(self):
            assert Validator().between(1, "TEST_KEY", 1, 100)

        def test_item_not_in_range_should_return_false(self):
            assert not Validator().between(0, "TEST_KEY", 1, 100)

    class TestGreaterThan:
        def test_none_should_return_false(self):
            assert not Validator().greater_than(None, "TEST_KEY", 1)

        def test_item_greater_than_should_return_true(self):
            assert Validator().greater_than(1.1, "TEST_KEY", 1)

        def test_item_equal_to_should_return_false(self):
            assert not Validator().greater_than(1, "TEST_KEY", 1)

    class TestLessThanOrEqualTo:
        def test_none_should_return_false(self):
            assert not Validator().less_than_or_equal_to(None, "TEST_KEY", 1)

        def test_item_equal_to_should_return_true(self):
            assert Validator().less_than_or_equal_to(1, "TEST_KEY", 1)

        def test_item_greater_than_should_return_false(self):
            assert not Validator().less_than_or_equal_to(1.1, "TEST_KEY", 1)

    class TestMatch:
        def test_none_should_return_false(self):
            assert not Validator().match(None, "TEST_KEY", r"^[0-9a-zA-Z]{1,63}$")

        def test_match_item_should_return_true(self):
            assert Validator().match("aA123bZz", "TEST_KEY", r"^[0-9a-zA-Z]{1,63}$")

        def test_not_match_item_should_return_false(self):
            assert not Validator().match("あiuえお", "TEST_KEY", r"^[0-9a-zA-Z]{1,63}$")

    class TestConstraint:
        def test_none_should_return_false(self):
            assert not Validator().constraint(None, "TEST_KEY", True, "None")

        def test_true_case_should_return_true(self):
            assert Validator().constraint("hoge", "TEST_KEY", True, "hoge")

        def test_false_case_should_return_false(self):
            assert not Validator().constraint("hoge", "TEST_KEY", False, "hoge")

    class TestExistDir:
        def test_none_should_return_false(self):
            assert not Validator().exist_dir(None, "TEST_KEY")

        def test_empty_path_should_return_false(self):
            assert not Validator().exist_dir("", "TEST_KEY")

        def test_exist_dir_path_should_return_true(self):
            dir_path = os.path.dirname(__file__)
            assert Validator().exist_dir(dir_path, "TEST_KEY")

        def test_exist_file_path_should_return_false(self):
            assert not Validator().exist_dir(__file__, "TEST_KEY")

        def test_non_exist_dir_path_should_return_false(self):
            dir_path = "/non/exist/dir/path"
            # 実際には存在しないディレクトリ名であることを確認
            # Confirm that the directory name does not actually exist
            assert not os.path.isdir(dir_path)
            assert not Validator().exist_dir(dir_path, "TEST_KEY")
