import paho.mqtt.publish as publish
from topic_type import TopicType

from raspi_web.common import config

HOST = config.get_str("MQTT_BROKER_HOST")
PORT = config.get_int("MQTT_BROKER_PORT")


def _publish_message(topic, payload):

    publish.single(
        topic=topic,
        payload=payload,
        hostname=HOST,
        port=PORT,
        retain=True,
    )


# ロガー
def publish_logger_status(logger_id: str):
    topic = f"logger/{logger_id}"

    _publish_message(topic=topic, payload=None)


def publish_logger_error(logger_id: str):
    topic = f"logger/{logger_id}/error"

    _publish_message(topic=topic, payload=None)


# センサー
def publish_sensor_status(logger_id: str, model: str, serial: str):
    topic = f"logger/{logger_id}/sensor/{model}/{serial}"

    _publish_message(topic=topic, payload=None)


def publish_sensor_error(logger_id: str, model: str, serial: str):
    topic = f"logger/{logger_id}/sensor/{model}/{serial}/error"

    _publish_message(topic=topic, payload=None)


def publish_sensor_loss(logger_id: str, model: str, serial: str):
    topic = f"logger/{logger_id}/sensor/{model}/{serial}/loss"

    _publish_message(topic=topic, payload=None)


def publish_sensor_abnormal(logger_id: str, model: str, serial: str):
    topic = f"logger/{logger_id}/sensor/{model}/{serial}/abnormal"

    _publish_message(topic=topic, payload=None)


# ハードウェアモニター
def publish_hwmonitor_status(logger_id: str):
    topic = f"hwmonitor/{logger_id}"

    _publish_message(topic=topic, payload=None)


def publish_hwmonitor_data(logger_id: str):
    topic = f"hwmonitor/{logger_id}/data"

    _publish_message(topic=topic, payload=None)


def publish_hwmonitor_error(logger_id: str):
    topic = f"hwmonitor/{logger_id}/error"

    _publish_message(topic=topic, payload=None)


# VCデータ
def publish_vc_data_level(logger_id: str, model: str, serial: str):
    topic = f"logger/{logger_id}/sensor/{model}/{serial}/vc"

    _publish_message(topic=topic, payload=None)


def publish_vc_data_fft(logger_id: str, model: str, serial: str):
    topic = f"logger/{logger_id}/sensor/{model}/{serial}/fft"

    _publish_message(topic=topic, payload=None)


def main(logger_id: str, model: str, serial: str, topic_type: TopicType):  # noqa: C901
    """指定したトピックにNoneを送り、メッセージを削除する

    Args:
        logger_id (str): サンプルトピックに用いるロガーID
        model (str): サンプルトピックに用いるセンサーモデル
        serial (str): サンプルトピックに用いるセンサーシリアル
        topic_type (TopicType): トピックの種類
    """
    match topic_type:
        # ロガー
        case "logger_status":
            publish_logger_status(logger_id=logger_id)  # type: ignore

        case "logger_error":
            publish_logger_error(logger_id=logger_id)  # type: ignore

        # センサー
        case "sensor_status":
            publish_sensor_status(logger_id=logger_id, model=model, serial=serial)  # type: ignore

        case "sensor_error":
            publish_sensor_error(logger_id=logger_id, model=model, serial=serial)  # type: ignore

        case "sensor_loss":
            publish_sensor_loss(logger_id=logger_id, model=model, serial=serial)

        case "sensor_abnormal":
            publish_sensor_abnormal(logger_id=logger_id, model=model, serial=serial)

        # ハードウェアモニター
        case "hwmonitor_status":
            publish_hwmonitor_status(logger_id=logger_id)  # type: ignore

        case "hwmonitor_data":
            publish_hwmonitor_data(logger_id=logger_id)  # type: ignore

        case "hwmonitor_error":
            publish_hwmonitor_error(logger_id=logger_id)  # type: ignore

        # VCデータ
        case "vc_data_level":
            publish_vc_data_level(logger_id=logger_id, model=model, serial=serial)

        case "vc_data_fft":
            publish_vc_data_fft(logger_id=logger_id, model=model, serial=serial)


if __name__ == "__main__":
    logger_id = "0"
    model = "A352"
    serial = "0001"

    topic_type: TopicType = "logger_status"

    main(logger_id=logger_id, model=model, serial=serial, topic_type=topic_type)
