# Copyright 2025, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

import logging
import sys

import dash
from dash import html

from raspi_web.common.config import get_str
from raspi_web.common.logger import init_logging, reduce_web_logging
from raspi_web.infra.subscriber import MQTTSubscriber


def main():
    # ロガーを設定する
    # Set up the logger
    _logger: logging.Logger = logging.getLogger(__name__.split(".")[-1])

    mqtt_subscriber: MQTTSubscriber | None = None
    try:
        # logging 初期化
        # Initialize logging
        init_logging()

        # Web サーバーからのログ出力を減らす
        # Reduce log output from the web server
        reduce_web_logging()

        # Dash 初期化
        # Initialize Dash
        app = dash.Dash(
            __name__,
            use_pages=True,
            pages_folder="ui/pages",
            update_title="",
            suppress_callback_exceptions=True,
        )
        app.layout = html.Div(dash.page_container)

        # MQTT 初期化
        # Initialize MQTT
        mqtt_subscriber = MQTTSubscriber()
        try:
            mqtt_subscriber.start()
        except ConnectionRefusedError:
            _logger.error(
                "Connection refused. Please check if message broker is running."
            )
            return 1

        # Dash 開始
        # Start Dash
        app.run(host=get_str("APP_HOST"), port=get_str("APP_PORT"), debug=False)

        return 0

    except Exception:
        _logger.exception("Unknown error occurred.")
        return 1

    finally:
        # MQTT 終了
        # End MQTT
        if mqtt_subscriber:
            mqtt_subscriber.end()


if __name__ == "__main__":
    code = main()
    sys.exit(code)
