# Copyright 2025, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

import os
from typing import List

import dotenv

# モジュールロードで環境変数を自動読み込み
# Automatically load environment variables when the module is loaded
dotenv.load_dotenv(override=True)


def init(env: str | None = None) -> None:
    """
    明示的 Dotenv 初期化

    通常はモジュールロードと同時にデフォルトの .env がロードされる想定

    テストなどの特殊な用途の初期化用

    Explicit Dotenv initialization

    Normally, the default .env is expected to be loaded when the module is loaded

    For special initialization purposes such as testing

    Args:
        env: .env ファイルへのパス, Optional
                Path to the .env file, Optional
    """
    dotenv.load_dotenv(env, override=True)


def get_str(name: str) -> str:
    """
    文字列データの取得

    Retrieve string data

    Args:
        name: 取得対象の環境変数名
                Name of the environment variable to retrieve
    """
    return os.getenv(name)  # type: ignore


def get_int(name: str) -> int:
    """
    数値データ(integer型)の取得

    Retrieve integer data

    Args:
        name: 取得対象の環境変数名
                Name of the environment variable to retrieve
    """
    val = get_str(name=name)

    return int(val)


def get_float(name: str) -> float:
    """
    数値データ(float型)の取得

    Retrieve float data

    Args:
        name: 取得対象の環境変数名
                Name of the environment variable to retrieve
    """
    val = get_str(name=name)

    return float(val)


def get_list(name: str) -> List[str]:
    """
    カンマ区切り文字列をカンマで分解しリスト取得

    Retrieve a list by splitting a comma-separated string

    Args:
        name: 取得対象の環境変数名
                Name of the environment variable to retrieve
    """
    val = get_str(name)

    return val.split(",")
