# Copyright 2025, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

import logging
import sys

from .config import get_str

# ログレベルを取得する
# Get the log level
_LOG_LEVEL: str = get_str("LOG_LEVEL")


class _Formatter(logging.Formatter):
    """
    ログ出力のフォーマッター

    Formatter for log output
    """
    def format(self, rec: logging.LogRecord) -> str:
        """
        ログのフォーマット用の関数

        Function for formatting logs

        Args:
            rec: ロギングされているイベントに含まれるすべての情報
                    All information contained in the logging event

        Returns:
            str: フォーマットされたログ
                    Formatted log
        """
        return "%-8s%s %s: %s" % (
            rec.levelname,
            self.formatTime(rec),
            rec.name,
            rec.getMessage(),
        )


def init_logging():
    """
    logging の初期化

    Initialization of logging
    """
    root = logging.getLogger()
    root.setLevel(_LOG_LEVEL)

    stream_h = logging.StreamHandler(sys.stdout)
    stream_h.setFormatter(_Formatter())
    logging.basicConfig(handlers=[stream_h], level=_LOG_LEVEL)


def reduce_web_logging():
    """
    Dash が内包する Web サーバーからのログ出力を減らす

    Reduce log output from the web server embedded in Dash
    """
    logging.getLogger("werkzeug").setLevel(logging.WARNING)
