# Copyright 2025, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

from dataclasses import dataclass, field

from .hwmonitor import HardwareMonitor
from .logger import Logger
from .sensor import Sensor


@dataclass
class DataSet:
    """
    特定のlogger_idのロガー、センサー、ハードウェアを保持するクラス

    A class that holds the logger, sensors, and hardware monitor for a specific logger_id

    Attributes:
        logger (Logger | None): ロガー. デフォルトはNone.
                Logger. Defaults to None.
        sensors (list[Sensor]): センサー. デフォルトは[].
                Sensors. Defaults to [].
        hwmonitor (HardwareMonitor): ハードウェアモニター. デフォルトはNone
                hwmonitor (HardwareMonitor): Hardware monitor. Defaults to None
    """
    logger: Logger | None = None
    sensors: list[Sensor] = field(default_factory=list)
    hwmonitor: HardwareMonitor | None = None

    def is_empty(self) -> bool:
        """
        データセットが空かどうかを判断する

        Determine if the dataset is empty

        Returns:
            bool: 空ならTrue, 1箇所でも値があればFalse
                    True if empty, False if any element has a value
        """
        if not (self.logger or self.sensors or self.hwmonitor):
            return True
        return False

    # ロガー関連のメソッド
    # Methods related to logger
    def get_logger(self, logger_id: str) -> Logger:
        """
        DataSetのロガーを返す

        DataSetになければ作成し返す

        Return the logger of the DataSet

        If not present, create and return it

        Args:
            logger_id (str): ロガーID
                    Logger ID

        Returns:
            Logger: DataSetのロガー
                    The logger of the DataSet
        """
        if self.logger is None:
            self.logger = Logger(logger_id=logger_id)

        return self.logger

    def clean_logger(self):
        """
        ロガーが空ならデータセットからロガーを削除

        Remove the logger from the dataset if it is empty
        """
        # ロガーがなければ何もしない
        # Do nothing if there is no logger
        if self.logger is None:
            return

        # ロガーが空なら削除
        # Remove the logger if it is empty
        if self.logger.is_empty():
            self.logger = None

    # センサー関連のメソッド
    # Methods related to sensors
    def _search_sensor(self, model: str, serial: str) -> Sensor | None:
        """
        センサー一覧からモデル, シリアル番号が一致するセンサーを返す

        Return the sensor from the list that matches the model and serial number

        Args:
            model (str): 探したいセンサーのモデル
                    The model of the sensor to search for
            serial (str): 探したいセンサーのシリアル番号
                    The serial number of the sensor to search for

        Returns:
            Sensor | None: 該当のセンサーがあればセンサー. なければNone
                    The matching sensor if found, otherwise None
        """
        for sensor in self.sensors:
            if sensor.model == model and sensor.serial == serial:
                return sensor
        return None

    def get_sensor(self, logger_id: str, model: str, serial: str) -> Sensor:
        """
        DataSetのセンサーを返す

        DataSetになければ作成し返す

        Return the sensor of the DataSet

        If not present, create and return it

        Args:
            logger_id (str): ロガーID
                    Logger ID
            model (str): 登録したいセンサーのモデル
                    The model of the sensor to register
            serial (str): 登録したいセンサーのシリアル
                    The serial number of the sensor to register

        Returns:
            Sensor: 該当のセンサー
                    The matching sensor
        """
        sensor = self._search_sensor(model=model, serial=serial)

        if sensor is None:
            sensor = Sensor(logger_id=logger_id, model=model, serial=serial)
            self.sensors.append(sensor)

        return sensor

    def clean_sensor(self, model: str, serial: str):
        """
        センサーが空ならデータセットからセンサーを削除

        Remove the sensor from the dataset if it is empty

        Args:
            model (str): センサーのモデル
                    The model of the sensor
            serial (str): センサーのシリアル
                    The serial number of the sensor
        """
        sensor = self._search_sensor(model=model, serial=serial)

        # 該当センサーが空なら削除
        # Remove the sensor if it is empty
        if sensor is not None and sensor.is_empty():
            self.sensors.remove(sensor)

    # ハードウェアモニター関連のメソッド
    # Methods related to hardware monitor
    def get_hwmonitor(self, logger_id: str) -> HardwareMonitor:
        """
        DataSetのハードウェアモニターを返す

        DataSetになければ作成し返す

        Returns the hardware monitor of the DataSet

        If not present, creates and returns it

        Args:
            logger_id (str): ロガーID
                    Logger ID

        Returns:
            HardwareMonitor: DataSetのハードウェアモニター
                    The hardware monitor of the DataSet
        """
        if self.hwmonitor is None:
            self.hwmonitor = HardwareMonitor(logger_id=logger_id)

        return self.hwmonitor

    def clean_hwmonitor(self):
        """
        ハードウェアモニターが空ならデータセットからハードウェアモニターを削除

        Remove the hardware monitor from the dataset if it is empty
        """
        # ハードウェアモニターがなければ何もしない
        # Do nothing if there is no hardware monitor
        if self.hwmonitor is None:
            return

        # ハードウェアモニターが空なら削除
        # Remove the hardware monitor if it is empty
        if self.hwmonitor.is_empty():
            self.hwmonitor = None
