# Copyright 2025, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

from collections import deque
from dataclasses import dataclass, field
from typing import Literal

from raspi_web.utils.constants import _MAX_MESSAGE_COUNT
from raspi_web.utils.types.error import ErrorType


@dataclass
class SensorStatus:
    """
    センサーのステータス

    Sensor status

    Attributes:
        timestamp (str): センサーの状態更新日時. "yyyy/mm/dd hh:mm:ss.mmmmmm".
                Timestamp of the sensor status update. "yyyy/mm/dd hh:mm:ss.mmmmmm".
        status (Literal["OK", "NG"]): センサーのステータス
                Sensor status
    """
    timestamp: str
    status: Literal["OK", "NG"]


@dataclass
class SensorError:
    """
    センサーのエラー

    Sensor error

    Attributes:
        timestamp (str): エラー発生日時. "yyyy/mm/dd hh:mm:ss.mmmmmm".
                Timestamp of the error occurrence. "yyyy/mm/dd hh:mm:ss.mmmmmm".
        level (ErrorType): エラーの深刻度
                Error severity
        message (str): エラーのメッセージ
                Error message
    """
    timestamp: str
    level: ErrorType
    message: str


@dataclass
class SensorMessage:
    """
    センサーに関するメッセージ

    Message related to the sensor

    Attributes:
        timestamp (str): メッセージ発生日時. "yyyy/mm/dd hh:mm:ss.mmmmmm".
                Timestamp of the message occurrence. "yyyy/mm/dd hh:mm:ss.mmmmmm".
        message (str): メッセージ
                Message content
    """
    timestamp: str
    message: str


@dataclass
class Sensor:
    """
    センサーの情報

    Sensor information

    Attributes:
        logger_id (str): センサーが紐づいているロガーのID
                Logger ID associated with the sensor
        model (str): センサーのモデル
                Sensor model
        serial (str): センサーのシリアル番号
                Sensor serial number
        status (SensorStatus): センサーの状態. デフォルトはNone.
                Sensor Status. Defaults to None.
        error (deque[SensorError]): センサーのエラー. 最大3件まで蓄積. 古いものから削除. デフォルトは[].
                Sensor errors. Stores up to 3 errors. Removes oldest first. Defaults to [].
        loss (deque[SensorMessage]): センサーのデータ欠落情報. 最大3件まで蓄積. 古いものから削除. デフォルトは[].
                Sensor loss. Stores up to 3 losses. Removes oldest first. Defaults to [].
        abnormal (deque[SensorError]): センサーの異常検知情報. 最大3件まで蓄積. 古いものから削除. デフォルトは[].
                Sensor abnormality. Stores up to 3 abnormalities. Removes oldest first. Defaults to [].
    """
    logger_id: str
    model: str
    serial: str
    status: SensorStatus | None = None
    error: deque[SensorError] = field(
        default_factory=lambda: deque(maxlen=_MAX_MESSAGE_COUNT)
    )
    loss: deque[SensorMessage] = field(
        default_factory=lambda: deque(maxlen=_MAX_MESSAGE_COUNT)
    )
    abnormal: deque[SensorMessage] = field(
        default_factory=lambda: deque(maxlen=_MAX_MESSAGE_COUNT)
    )

    def update_status(self, status: SensorStatus | None):
        """
        センサーのステータスを更新する

        Update the sensor status

        Args:
            status (SensorStatus | None): 更新したい値
                    Value to update
        """
        self.status = status

    def update_error(self, error: SensorError | None):
        """
        センサーのエラーを更新する

        Update the sensor errors

        Args:
            error (SensorError | None): 更新したい値
                    Value to update
        """
        if error is not None:
            self.error.append(error)
        else:
            self.error = deque[SensorError](maxlen=_MAX_MESSAGE_COUNT)

    def update_loss(self, loss: SensorMessage | None):
        """
        センサーのデータ欠落情報を更新する

        Update the sensor losses

        Args:
            loss (SensorMessage | None): 更新したい値
                    Value to update
        """
        if loss is not None:
            self.loss.append(loss)
        else:
            self.loss = deque[SensorMessage](maxlen=_MAX_MESSAGE_COUNT)

    def update_abnormal(self, abnormal: SensorMessage | None):
        """
        センサーの異常検知情報を更新する

        Update the sensor abnormalities

        Args:
            abnormal (SensorMessage | None): 更新したい値
                    Value to update
        """
        if abnormal is not None:
            self.abnormal.append(abnormal)
        else:
            self.abnormal = deque[SensorMessage](maxlen=_MAX_MESSAGE_COUNT)

    def is_empty(self) -> bool:
        """
        センサーの各要素が空かどうかを判断する

        Check if all elements of the sensor are empty

        Returns:
            bool: 空ならTrue, 1箇所でも値があればFalse
                    True if empty, False if any element has a value
        """
        return not (self.status or self.error or self.loss or self.abnormal)
