# Copyright 2025, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

from typing import Union

import dash
from dash import dash_table, html
from dash.dependencies import MATCH, Input, Output, State

from raspi_web.data.data_set import DataSet
from raspi_web.data.hwmonitor import HardwareMonitor
from raspi_web.data.logger import Logger
from raspi_web.data.sensor import Sensor

# Dashアプリケーションのインスタンスを取得する
# Get the instance of the Dash application
app = dash.get_app()

# ロガーテーブルの列定義
# Logger table columns definition
_LOGGER_TABLE_COLUMNS = [
    {"id": "status", "name": "Status"},
    {"id": "error", "name": "Error"},
]

# センサーテーブルの列定義
# Sensor table columns definition
_SENSOR_TABLE_COLUMNS = [
    {"id": "model", "name": "Model"},
    {"id": "serial", "name": "Serial number"},
    {"id": "status", "name": "Status"},
    {"id": "graph_link", "name": "VC judgment value\nand velocity graph"},
    {"id": "error", "name": "Error"},
    {"id": "loss_message", "name": "Data loss"},
    {"id": "abnormal_message", "name": "Element abnormality"},
]

# ハードウェアモニターテーブルの列定義
# Hardware monitor table columns definition
_HARDWAREMONITOR_TABLE_COLUMNS = [
    {"id": "status", "name": "Status"},
    {"id": "cpu_temperature", "name": "CPU temperature\n(degC)"},
    {"id": "cpu_usage", "name": "CPU usage\n(%)"},
    {"id": "memory_usage", "name": "Memory usage\n(%)"},
    {"id": "disk_usage", "name": "Disk usage\n(%)"},
    {"id": "error", "name": "Error"},
]


def _create_logger_display_data(logger: Logger) -> dict:
    """
    テーブル表示のためにロガーをdictに加工する

    Process logger into a dict for table display

    Args:
        logger (Logger): 加工したいロガー
                Logger to be processed

    Returns:
        dict: 加工したロガーデータ
                Processed logger data
    """
    logger_status = logger.status.status if logger.status else "None"
    logger_error: str = ""

    if logger.error:
        for error in logger.error:
            logger_error += f"{error.level}: {error.message}\n"
    else:
        logger_error = "None"

    logger_display_data = {
        "status": logger_status,
        "error": logger_error,
    }
    return logger_display_data


def _create_sensor_display_data(sensor: Sensor) -> dict:
    """
    テーブル表示のためにロガーをdictに加工する

    Process sensor into a dict for table display

    Args:
        sensor (Sensor): 加工したいセンサー
                Sensor to be processed

    Returns:
        dict: 加工したセンサーデータ
                Processed sensor data
    """
    model = sensor.model
    serial = sensor.serial
    sensor_status = sensor.status.status if sensor.status else "None"
    sensor_error: str = ""
    if sensor.error:
        for error in sensor.error:
            sensor_error += f"{error.level}: {error.message}\n"
    else:
        sensor_error = "None"

    sensor_loss_message: str = ""
    if sensor.loss:
        for loss in sensor.loss:
            sensor_loss_message += f"{loss.message}\n"
    else:
        sensor_loss_message = "None"

    sensor_abnormal_message: str = ""
    if sensor.abnormal:
        for abnormal in sensor.abnormal:
            sensor_abnormal_message += f"{abnormal.message}\n"
    else:
        sensor_abnormal_message = "None"

    sensor_display_data = {
        # logger_idはグラフ表示のための値でテーブルには表示しない
        # logger_id is a value for graph display, not shown in table
        "logger_id": sensor.logger_id,
        "model": model,
        "serial": serial,
        "status": sensor_status,
        "graph_link": "Open graph",
        "error": sensor_error,
        "loss_message": sensor_loss_message,
        "abnormal_message": sensor_abnormal_message,
    }
    return sensor_display_data


def _create_hwmonitor_display_data(hwmonitor: HardwareMonitor) -> dict:
    """
    テーブル表示のためにハードウェアモニターをdictに加工する

    Process hardware monitor into a dict for table display

    Args:
        hwmonitor (HardwareMonitor): 加工したいハードウェアモニター
                Hardware monitor to be processed

    Returns:
        dict: 加工したハードウェアモニター
                Processed hardware monitor
    """
    hwmonitor_status = hwmonitor.status.status if hwmonitor.status else "None"

    cpu_temperature: Union[float, str] = "None"
    cpu_usage: Union[float, str] = "None"
    memory_usage: Union[float, str] = "None"
    disk_usage: Union[float, str] = "None"
    if hwmonitor.data:
        cpu_temperature = hwmonitor.data.cpu_temperature
        cpu_usage = hwmonitor.data.cpu_usage
        memory_usage = hwmonitor.data.memory_usage
        disk_usage = hwmonitor.data.disk_usage

    hwmonitor_error: str = ""
    if hwmonitor.error:
        for error in hwmonitor.error:
            hwmonitor_error += f"{error.level}: {error.message}\n"
    else:
        hwmonitor_error = "None"

    display_data = {
        "status": hwmonitor_status,
        "cpu_temperature": cpu_temperature,
        "cpu_usage": cpu_usage,
        "memory_usage": memory_usage,
        "disk_usage": disk_usage,
        "error": hwmonitor_error,
    }
    return display_data


def data_set_component(logger_id: str, data_set: DataSet) -> html.Div:
    """
    データセットの情報を表示するコンポーネントを返す

    Returns a component that displays dataset information

    Args:
        logger_id: ロガーID
                Logger ID
        data_set (DataSet): コンポーネントを作成するDataSet
                DataSet to create the component

    Returns:
        html.Div: ロガーID、ロガー情報を表示するテーブルを含んだDivコンポーネント
                Div component containing tables displaying logger ID and logger information
    """

    # ロガーデータ生成
    # Generate logger data
    logger_display = html.Div("No logger information available")
    if data_set.logger:
        logger_data = _create_logger_display_data(logger=data_set.logger)
        logger_display = dash_table.DataTable(
            id={
                "type": "logger_table",
                "index": logger_id,
            },
            columns=_LOGGER_TABLE_COLUMNS,
            data=[logger_data],
            css=[{"rule": "table-layout: fixed"}],
            # カラムごとの横幅は項目名、項目の内容に合わせ調整
            # Adjust column widths to match item names and contents
            style_cell_conditional=[
                {"if": {"column_id": "status"}, "width": "22%"},
                {"if": {"column_id": "error"}, "width": "78%"},
            ],
            style_table={"width": "28.8%"},
            style_cell={
                "whiteSpace": "pre-line",
                "textAlign": "left",
                "wordBreak": "break-word",
            },
        )

    # センサーデータ生成
    # Generate sensor data
    sensor_display = html.Div("No sensor information available")
    if data_set.sensors:
        sensors_data = [
            _create_sensor_display_data(sensor=sensor) for sensor in data_set.sensors
        ]
        sensor_display = dash_table.DataTable(
            id={
                "type": "sensor_table",
                "index": logger_id,
            },
            columns=_SENSOR_TABLE_COLUMNS,
            data=sensors_data,
            css=[{"rule": "table-layout: fixed"}],
            # カラムごとの横幅は項目名、項目の内容に合わせ調整
            # Adjust column widths to match item names and contents
            style_cell_conditional=[
                {"if": {"column_id": "model"}, "width": "8%"},
                {"if": {"column_id": "serial"}, "width": "8%"},
                {"if": {"column_id": "status"}, "width": "8%"},
                {"if": {"column_id": "graph_link"}, "width": "10%"},
                {"if": {"column_id": "error"}, "width": "30%"},
                {"if": {"column_id": "loss_message"}, "width": "18%"},
                {"if": {"column_id": "abnormal_message"}, "width": "18%"},
            ],
            # グラフボタンに色付け
            # Color the graph button
            style_data_conditional=[
                {
                    "if": {"column_id": "graph_link"},
                    "color": "#0b5aff",
                    "textDecoration": "underline",
                }
            ],
            style_table={"width": "80%"},
            style_cell={
                "whiteSpace": "pre-line",
                "textAlign": "left",
                "wordBreak": "break-word",
            },
        )

    # ハードウェアモニターデータ生成
    # Generate hardware monitor data
    hwmonitor_display = html.Div("No hardware information available")
    if data_set.hwmonitor:
        hwmonitor_data = _create_hwmonitor_display_data(hwmonitor=data_set.hwmonitor)
        hwmonitor_display = dash_table.DataTable(
            id={
                "type": "hwmonitor_table",
                "index": logger_id,
            },
            columns=_HARDWAREMONITOR_TABLE_COLUMNS,
            data=[hwmonitor_data],
            css=[{"rule": "table-layout: fixed"}],
            # カラムごとの横幅は項目名、項目の内容に合わせ調整
            # Adjust column widths to match item names and contents
            style_cell_conditional=[
                {"if": {"column_id": "status"}, "width": "11%"},
                {"if": {"column_id": "cpu_temperature"}, "width": "11%"},
                {"if": {"column_id": "cpu_usage"}, "width": "11%"},
                {"if": {"column_id": "memory_usage"}, "width": "11%"},
                {"if": {"column_id": "disk_usage"}, "width": "11%"},
                {"if": {"column_id": "error"}, "width": "45%"},
            ],
            style_table={"width": "54.4%"},
            style_cell={
                "whiteSpace": "pre-line",
                "textAlign": "left",
                "wordBreak": "break-word",
            },
        )

    return html.Div(
        [
            html.H2(f"Logger ID: {logger_id}"),
            html.H3("Logger Program Status"),
            logger_display,
            html.H3("Sensor Status"),
            sensor_display,
            html.H3("Raspberry Pi Hardware Status"),
            hwmonitor_display,
            html.Div(id={"type": "dummy-div", "index": logger_id}),
        ]
    )


# グラフ画面遷移
# Graph screen transition
app.clientside_callback(
    """
    function(active_cell, data) {
        if (active_cell["column_id"] == "graph_link"){
            const row = active_cell["row"];
            const sensor = data[row]
            loggerID = sensor["logger_id"]
            model = sensor["model"]
            serial = sensor["serial"]
            const currentUrl = window.location.href;
            const url = `${currentUrl}graph?logger_id=${loggerID}&model=${model}&serial=${serial}`;
            window.open(url, '_blank');
        }
    }
    """,
    Output({"type": "dummy-div", "index": MATCH}, "children"),
    Input({"type": "sensor_table", "index": MATCH}, "active_cell"),
    State({"type": "sensor_table", "index": MATCH}, "data"),
    prevent_initial_call=True,  #
)
