# Copyright 2025, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

from dataclasses import dataclass
from typing import Any


@dataclass
class SensorSummary:
    """
    センサーのサマリー情報

    Sensor summary information

    Attributes:
        summary (dict): センサーのサマリー
                Sensor summary
    """

    summary: dict | None = None

    def update_summary(self, summary: dict | None):
        """
        センサーのサマリーを更新する

        Update the sensor summary

        Args:
            summary (dict | None): 更新したい値
                    Value to update
        """
        self.summary = summary

    def is_empty(self) -> bool:
        """
        センサーサマリーの要素が空かどうかを判断する

        Check if all elements of the sensor summary are empty

        Returns:
            bool: 空ならTrue, 1箇所でも値があればFalse
                    True if empty, False if any element has a value
        """
        return not self.summary

    def __getitem__(self, key: str) -> Any:
        """
        SensorSummary が保持するデータ辞書に外部から `[]` でアクセスできるようにする特殊関数

        Special method to allow external access to the data dictionary in SensorSummary
        using `[]`

        Args:
            key (str): データ辞書のキー
                Key of the data dictionary

        Returns:
            Any: 辞書の要素
                Element of the dictionary

        Raises:
            KeyError: 辞書にないキーが指定された場合
                Raised when a key not present in the dictionary is specified
        """
        if self.summary:
            return self.summary[key]
        else:
            raise KeyError(key)
