# Copyright 2025, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

import logging

import paho.mqtt.client as mqtt
from paho.mqtt.enums import CallbackAPIVersion

from raspi_web.util import config

from .data_manager import DataManager
from .topic import HARDWAREMONITOR_TOPIC_BASE, LOGGER_TOPIC_BASE, SUMMARY_TOPIC_BASE


class MQTTSubscriber:
    # ロガーを設定する
    # Set up the logger
    _logger: logging.Logger = logging.getLogger(__name__.split(".")[-1])

    @staticmethod
    def _on_connect(client: mqtt.Client, userdata, flags, reason_code, properties):
        """
        トピックのサブスクライブ対象トピックを設定

        Set the topics to subscribe to
        """
        MQTTSubscriber._logger.info("Connected to message broker")
        MQTTSubscriber._logger.debug(f"At on_connect: flags={flags}, rc={reason_code}")

        # 対象トピック
        # Target topics
        topics = [
            (f"{LOGGER_TOPIC_BASE}/#", 0),
            (f"{HARDWAREMONITOR_TOPIC_BASE}/#", 0),
            (f"{SUMMARY_TOPIC_BASE}/#", 0),
        ]

        # サブスクライブ
        # Subscribe
        client.subscribe(topic=topics)

    @staticmethod
    def _on_message(client: mqtt.Client, userdata, msg: mqtt.MQTTMessage):
        """
        ブローカーから受け取ったトピック、メッセージをDataManagerへ通知

        Notify DataManager of the topic and message received from the broker
        """
        MQTTSubscriber._logger.info(
            "{}{}: {}".format(
                msg.topic, " (retain)" if msg.retain else "", str(msg.payload)
            )
        )

        data_manager = DataManager.get_instance()
        data_manager.receive_notification(topic=msg.topic, payload=msg.payload)

    def __init__(self):
        """
        MQTTサブスクライバー

        MQTTブローカーをサブスクライブし, 受け取ったトピックとメッセージをDataManagerに通知する

        MQTT Subscriber

        Subscribe to the MQTT broker and notify DataManager of the received topics and messages
        """
        # Setup client
        self.mqtt_client = mqtt.Client(
            CallbackAPIVersion.VERSION2, client_id="mqtt_subscriber"
        )
        self.mqtt_client.on_connect = self._on_connect
        self.mqtt_client.on_message = self._on_message

    def start(self):
        """
        サブスクライブの開始

        Start subscribing
        """
        # ブローカーとの接続
        # Connect to the broker
        self.mqtt_client.connect(
            host=config.get_str("MQTT_BROKER_HOST"),
            port=config.get_int("MQTT_BROKER_PORT"),
        )

        # サブスクライブの開始
        # Start subscribing
        self.mqtt_client.loop_start()
        self._logger.info("---loop_start---")

    def end(self):
        """
        サブスクライブの終了

        End subscribing
        """
        self.mqtt_client.loop_stop()
        self._logger.info("---loop_end---")
