# Copyright 2025, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

import dash
from dash import dcc, html
from dash.dependencies import Input, Output

from raspi_web.infra.data_manager import DataManager
from raspi_web.ui.component.data_set import data_set_component

# Dashアプリケーションのインスタンスを取得
# Get the instance of the Dash application
app = dash.get_app()

# ルートパスにページを登録
# Register the page at the root path
dash.register_page(__name__, path="/")

# ホーム画面のレイアウトを定義
# Define the layout for the home screen
layout = html.Div(
    [
        html.H1("Logger List"),
        # 1秒ごとに更新（ミリ秒単位）
        # Update every second (in milliseconds)
        dcc.Interval(
            id="interval_component",
            interval=1 * 1000,
            n_intervals=0,
        ),
        # ロガーの表示
        # Display loggers
        html.Div(id="loggers_div"),
    ]
)


# インターバルで設定した秒毎に実行されるコールバック関数を定義
# Define the callback function to be executed at intervals set in seconds
@app.callback(
    Output("loggers_div", "children"), Input("interval_component", "n_intervals")
)
def update_table(n):
    """
    インターバルで設定した秒毎にDataManagerからロガーの一覧を取得し、コンポーネント化して返す

    Retrieve the list of loggers from DataManager at intervals set in seconds, and return as components

    Returns:
        list[html.Div]: ロガー一覧
                List of loggers
    """
    data_manager = DataManager.get_instance()

    # DataManager の複数のデータ取得メソッドを呼び出すため、ロックコンテキストを構成する
    # Use lock context to call multiple query methods of the DataManager
    with data_manager.lock:
        data_sets = data_manager.get_data_sets()

        # グラフ存在判定用キーセット
        # Set of keys with a graph data
        vc_data_keyset = data_manager.get_vc_data_keyset()
        summary_keyset = data_manager.get_summary_keyset()
        summary_alert = data_manager.get_summary_alert()

    data_set_components: list[html.Div | html.Hr] = []
    for logger_id in sorted(data_sets.keys()):
        data_set = data_sets[logger_id]
        # ロガー複数件の場合はセパレータ挿入
        # Put a separator if multiple loggers
        if data_set_components:
            data_set_components.append(html.Hr(style={"margin-top": "2em"}))

        # データセット要素を追加
        # Add a dataset component
        data_set_components.append(
            data_set_component(
                logger_id=logger_id,
                data_set=data_set,
                vc_data_keyset=vc_data_keyset,
                summary_keyset=summary_keyset,
                summary_alert=summary_alert,
            )
        )

    return data_set_components
