# Copyright 2025, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

import math
from urllib.parse import parse_qs, urlparse

import dash
import plotly.graph_objects as go
from dash import dcc, html
from dash.dependencies import Input, Output

from raspi_web.infra.data_manager import DataManager
from raspi_web.ui.component.target_sensor import target_sensor

# Dashアプリケーションのインスタンスを取得する
# Get the instance of the Dash application
app = dash.get_app()

# グラフページを登録する
# Register the graph page
dash.register_page(__name__, path_template="/vc")


# 一度だけオクターブバンドを計算
# Calculate octave bands once
def _calc_octave_bund(octave_rate: int, octave_limit_frequency: float) -> list[float]:
    """
    オクターブバンドを計算する関数

    Function to calculate octave bands

    Args:
        octave_rate (int): 1/octave_rateバンドを計算する
                Calculate 1/octave_rate bands
        octave_limit_frequency (float): オクターブバンドをどこの周波数まで計算するか
                Frequency limit up to which octave bands are

    Returns:
        list[float]: オクターブバンド計算結果
                Result of octave band calculation
    """
    center_frequency_list = []
    exponent = 0
    while pow(2, exponent / octave_rate) <= octave_limit_frequency:
        center_frequency_list.append(pow(2, exponent / octave_rate))
        exponent += 1
    return center_frequency_list


_BUFFERED_X_VALUES = _calc_octave_bund(
    octave_rate=3,
    octave_limit_frequency=500,
)

# 軸の表示範囲(ログスケール)
# Axis display range (log scale)
_X_AXIS_RANGE = [-0.1, 2.7]
_Y_AXIS_RANGE = [-5, 3]


# 軸の目盛りの表示を計算
# Calculate axis tick values
def _calc_tick_values(range_start: float, range_end: float) -> list[float]:
    """
    軸の目盛りを計算する関数

    指定された範囲の間の10のべき乗の値を返す

    Function to calculate axis tick values

    Returns powers of 10 within the specified range

    Args:
        range_start (float): 範囲の始まり
                Start of the range
        range_end (float): 範囲の終わり
                End of the range

    Returns:
        list[float]: 範囲内の10のべき乗のリスト
                List of powers of 10 within the range
    """
    tick_values = []
    # range_startからrange_endまでの10のべき乗の値を取得
    # Get powers of 10 from range_start to range_end
    for exponent in range(math.ceil(range_start), math.floor(range_end) + 1):
        tick_values.append(10**exponent)
    return tick_values


_X_AXIS_TICK_VALUES = _calc_tick_values(_X_AXIS_RANGE[0], _X_AXIS_RANGE[1])

## 小さい数値が1e-5のように計算されるため、小数点以下6桁まで0.00001のようにフォーマットを変換して軸のテキストを表示
## Since small values are calculated as 1e-5, format them to display axis text as 0.00001 up to 6 decimal places
_X_AXIS_TICK_TEXT = [f"{x:.6f}".rstrip("0").rstrip(".") for x in _X_AXIS_TICK_VALUES]

_Y_AXIS_TICK_VALUES = _calc_tick_values(_Y_AXIS_RANGE[0], _Y_AXIS_RANGE[1])

## 小さい数値が1e-5のように計算されるため、小数点以下6桁まで0.00001のようにフォーマットを変換して軸のテキストを表示
## Since small values are calculated as 1e-5, format them to display axis text as 0.00001 up to 6 decimal places
_Y_AXIS_TICK_TEXT = [f"{y:.6f}".rstrip("0").rstrip(".") for y in _Y_AXIS_TICK_VALUES]


# 軸の補助線を計算
# Calculate minor grid values
def _calc_minor_grid_values(range_start: float, range_end: float) -> list[float]:
    """
    補助線を引く値を計算する関数

    指定された範囲より広い範囲まで補助線を引く値を計算する

    Function to calculate values for minor grid lines

    Calculates values for a range wider than the specified range

    Args:
        range_start (float): 範囲の始まり
                Start of the range
        range_end (float): 範囲の終わり
                End of the range

    Returns:
        list[float]: 補助線を引く値のリスト
                List of values for minor grid lines
    """
    minor_grid_values = []
    ## 表示範囲より広めに補助線を引く位置を計算
    ## Calculate positions for minor grid lines slightly wider than the display range
    for exponent in range(math.floor(range_start), math.ceil(range_end)):
        base = 10**exponent
        minor_grid_values.extend([base * i for i in range(1, 10)])
    return minor_grid_values


_X_AXIS_MINOR_GRID_VALUES = _calc_minor_grid_values(
    range_start=_X_AXIS_RANGE[0], range_end=_X_AXIS_RANGE[1]
)
_Y_AXIS_MINOR_GRID_VALUES = _calc_minor_grid_values(
    range_start=_Y_AXIS_RANGE[0], range_end=_Y_AXIS_RANGE[1]
)


def layout(logger_id=None, model=None, serial=None, **other_unknown_query_strings):
    """
    VC グラフ画面のレイアウトを定義して返す関数

    Args:
        logger_id: ロガーID
                Logger ID
        model: モデル
                Model
        serial: シリアルNO
                Serial number

    Returns:
        html.Div: VC グラフ画面を表示するDIVコンポーネント
                DIV component displaying the VC graph screen
    """
    return (
        html.Div(
            [
                html.H1("VC Judgment Value and Velocity Graph"),
                target_sensor(logger_id, model, serial),
                html.Div(id="graph_area"),
                dcc.Location(id="url"),
                dcc.Interval(
                    id="graph_interval_component",
                    # 1秒ごとに更新（ミリ秒単位）
                    # Update every second (in milliseconds)
                    interval=1 * 1000,
                    n_intervals=0,
                ),
            ]
        )
        if logger_id and model and serial
        else None
    )


@app.callback(
    Output("graph_area", "children"),
    Input("url", "search"),
    Input("graph_interval_component", "n_intervals"),
)
def update_graph(search, n):
    """
    インターバルで設定した秒毎にDataManagerからVC判定値、速度データを取得し、グラフコンポーネント化して返す

    Retrieve VC judgment value and velocity data from DataManager at intervals set in seconds,
    and return as graph components

    Args:
        search: 表示しているページのURL
                URL of the page being displayed

    Returns:
        html.Div|None: VC判定値、速度グラフコンポーネント
                VC judgment value and velocity graph components
    """
    query_params_str = urlparse(search).query
    query_params_dict = parse_qs(query_params_str)
    logger_id = query_params_dict["logger_id"][0]
    model = query_params_dict["model"][0]
    serial = query_params_dict["serial"][0]

    data_manager = DataManager.get_instance()
    vc_data = data_manager.get_vc_data(logger_id=logger_id, model=model, serial=serial)
    if vc_data is None:
        return html.Div("No data available")

    if not (vc_data.vc_level and vc_data.fft_data):
        return html.Div("Data is being prepared")

    # VC判定レベル
    # VC judgment level
    vc_level_display = html.H2(f"VC Judgment Level: {vc_data.vc_level.level}")

    # 速度グラフ
    # Velocity graph
    y_values = vc_data.fft_data.value
    ## yの値の長さに合わせ、xの値を取得
    ## Get x values corresponding to the length of y values
    x_values = _BUFFERED_X_VALUES[: len(y_values)]

    x_axis_setting = go.layout.XAxis(
        title="Frequency (Hz)",
        type="log",
        range=_X_AXIS_RANGE,
        tickmode="array",
        tickvals=_X_AXIS_TICK_VALUES,
        ticktext=_X_AXIS_TICK_TEXT,
        gridcolor="#C8C8C8",
        linecolor="black",
        linewidth=1,
        minor={
            "tickvals": _X_AXIS_MINOR_GRID_VALUES,
            "showgrid": True,
            "gridcolor": "#C8C8C8",
        },
    )
    y_axis_setting = go.layout.YAxis(
        title="Velocity (mm/s)",
        type="log",
        range=_Y_AXIS_RANGE,
        tickmode="array",
        tickvals=_Y_AXIS_TICK_VALUES,
        ticktext=_Y_AXIS_TICK_TEXT,
        gridcolor="#C8C8C8",
        linecolor="black",
        linewidth=1,
        minor={
            "tickvals": _Y_AXIS_MINOR_GRID_VALUES,
            "showgrid": True,
            "gridcolor": "#C8C8C8",
        },
    )
    layout = go.Layout(
        plot_bgcolor="white",
        width=600,
        height=600,
        xaxis=x_axis_setting,
        yaxis=y_axis_setting,
        margin={"l": 0, "r": 0, "t": 20, "b": 0},
    )

    data = go.Scatter(
        mode="markers", marker={"color": "blue", "size": 8}, x=x_values, y=y_values
    )

    fig = go.Figure(data=data, layout=layout)

    return [
        vc_level_display,
        dcc.Graph(figure=fig),
    ]
