# Copyright 2025, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

from raspi_web.data.data_set import DataSet
from raspi_web.data.hwmonitor import HardwareMonitor, HardwareStatus
from raspi_web.data.logger import Logger, LoggerStatus
from raspi_web.data.sensor import Sensor, SensorStatus


class TestDataSet:
    def test_is_empty(self):
        data_set = DataSet()
        expect = True

        actual = data_set.is_empty()

        assert expect == actual

    def test_is_empty_exist_logger(self):
        logger = Logger(
            logger_id="0",
            status=LoggerStatus(timestamp="2000/11/11 11:11:11", status="START"),
        )

        data_set = DataSet(logger=logger)

        expect = False

        actual = data_set.is_empty()

        assert expect == actual

    def test_is_empty_exist_sensors(self):
        sensor = Sensor(logger_id="0", model="A352", serial="0001")
        data_set = DataSet(sensors=[sensor])

        expect = False

        actual = data_set.is_empty()

        assert expect == actual

    def test_is_empty_exist_hwmonitor(self):
        hwmonitor = HardwareMonitor(logger_id="0")

        data_set = DataSet(hwmonitor=hwmonitor)

        expect = False

        actual = data_set.is_empty()

        assert expect == actual

    # ロガー関連のテスト
    # Tests related to Logger
    def test_get_logger(self):
        logger_id = "0"
        expect_logger = Logger(logger_id="0")
        data_set = DataSet(logger=expect_logger)

        actual_logger = data_set.get_logger(logger_id=logger_id)

        assert id(expect_logger) == id(actual_logger)

    def test_get_logger_not_exist(self):
        logger_id = "0"
        data_set = DataSet()

        actual_logger = data_set.get_logger(logger_id=logger_id)

        assert logger_id == actual_logger.logger_id

    def test_clean_logger(self):
        logger = Logger(logger_id="0")
        data_set = DataSet(logger=logger)

        data_set.clean_logger()

        assert data_set.logger is None

    def test_clean_logger_not_empty(self):
        logger = Logger(
            logger_id="0",
            status=LoggerStatus(timestamp="2000/01/01 01:01:01", status="START"),
        )
        data_set = DataSet(logger=logger)

        data_set.clean_logger()

        assert data_set.logger is not None

    def test_clean_logger_is_none(self):
        data_set = DataSet()

        data_set.clean_logger()

        assert data_set.logger is None

    # センサー関連のテスト
    # Tests related to Sensor
    def test__search_sensor(self):
        model = "A352"
        serial = "0001"
        expect_sensor = Sensor(logger_id="0", model=model, serial=serial)
        data_set = DataSet(sensors=[expect_sensor])

        actual_sensor = data_set._search_sensor(model=model, serial=serial)

        assert id(expect_sensor) == id(actual_sensor)

    def test__search_sensor_is_none_not_match_model(self):
        model = "0"
        serial = "0001"
        sensor = Sensor(logger_id="0", model=model, serial=serial)
        data_set = DataSet(sensors=[sensor])
        actual = data_set._search_sensor(model="1", serial=serial)

        assert actual is None

    def test__search_sensor_is_none_not_match_serial(self):
        model = "0"
        serial = "0001"
        sensor = Sensor(logger_id="0", model=model, serial=serial)
        data_set = DataSet(sensors=[sensor])
        actual = data_set._search_sensor(model=model, serial="0002")

        assert actual is None

    def test_get_sensor(self):
        logger_id = "0"
        model = "A352"
        serial0 = "0000"
        serial1 = "0001"

        sensor0 = Sensor(logger_id=logger_id, model=model, serial=serial0)
        sensor1 = Sensor(logger_id=logger_id, model=model, serial=serial1)
        data_set = DataSet(sensors=[sensor0, sensor1])

        actual_sensor = data_set.get_sensor(
            logger_id=logger_id, model=model, serial=serial1
        )

        assert id(sensor1) == id(actual_sensor)

    def test_get_sensor_not_exist(self):
        logger_id = "0"
        model = "A352"
        serial = "0001"

        sensor = Sensor(logger_id=logger_id, model=model, serial=serial)
        data_set = DataSet(sensors=[sensor])

        expect_serial = "0002"
        _ = data_set.get_sensor(logger_id=logger_id, model=model, serial=expect_serial)

        assert len(data_set.sensors) == 2
        assert expect_serial == data_set.sensors[-1].serial

    def test_clean_sensor(self):
        logger_id = "0"
        model = "A352"
        serial = "0001"

        sensor = Sensor(logger_id=logger_id, model=model, serial=serial)
        data_set = DataSet(sensors=[sensor])

        data_set.clean_sensor(model=model, serial=serial)

        assert len(data_set.sensors) == 0

    def test_clean_sensor_not_empty(self):
        logger_id = "0"
        model = "A352"
        serial = "0001"

        sensor = Sensor(
            logger_id=logger_id,
            model=model,
            serial=serial,
            status=SensorStatus(timestamp="2000/01/01 01:01:01", status="OK"),
        )
        data_set = DataSet(sensors=[sensor])

        data_set.clean_sensor(model=model, serial=serial)

        assert len(data_set.sensors) == 1

    # ハードウェアモニター関連のテスト
    # Tests related to Hardware Monitor
    def test_get_hwmonitor(self):
        logger_id = "0"
        expect_hwmonitor = HardwareMonitor(logger_id="0")
        data_set = DataSet(hwmonitor=expect_hwmonitor)

        actual_hwmonitor = data_set.get_hwmonitor(logger_id=logger_id)

        assert id(expect_hwmonitor) == id(actual_hwmonitor)

    def test_get_hwmonitor_not_exist(self):
        logger_id = "0"
        data_set = DataSet()

        actual_hwmonitor = data_set.get_hwmonitor(logger_id=logger_id)

        assert logger_id == actual_hwmonitor.logger_id

    def test_clean_hwmonitor(self):
        hwmonitor = HardwareMonitor(logger_id="0")
        data_set = DataSet(hwmonitor=hwmonitor)

        data_set.clean_hwmonitor()

        assert data_set.hwmonitor is None

    def test_clean_hwmonitor_not_empty(self):
        hwmonitor = HardwareMonitor(
            logger_id="0",
            status=HardwareStatus(timestamp="2000/01/01 01:01:01", status="START"),
        )
        data_set = DataSet(hwmonitor=hwmonitor)

        data_set.clean_hwmonitor()

        assert data_set.hwmonitor is not None

    def test_clean_hwmonitor_is_none(self):
        data_set = DataSet()

        data_set.clean_hwmonitor()

        assert data_set.hwmonitor is None
