# Copyright 2025, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

from collections import deque
from typing import Literal

from raspi_web.const import _MAX_MESSAGE_COUNT
from raspi_web.data.error_type import ErrorType
from raspi_web.data.hwmonitor import (
    HardwareData,
    HardwareError,
    HardwareMonitor,
    HardwareStatus,
)


class TestHardwareMonitor:

    def test_update_status(self):
        hwmonitor = HardwareMonitor(logger_id="0")
        expect: Literal["START", "STOP"] = "START"
        update_info = HardwareStatus(timestamp="2000/11/11 11:11:11", status=expect)

        hwmonitor.update_status(status=update_info)

        assert hwmonitor.status is not None
        assert expect == hwmonitor.status.status

    def test_update_status_to_none(self):
        hwmonitor = HardwareMonitor(
            logger_id="0",
            status=HardwareStatus(timestamp="2000/11/11 11:11:11", status="START"),
        )
        update_info = None

        hwmonitor.update_status(status=update_info)

        assert hwmonitor.status is None

    def test_update_data(self):
        hwmonitor = HardwareMonitor(logger_id="0")
        expect_cpu_temperature = 10.1
        expect_cpu_usage = 10.2
        expect_memory_usage = 10.3
        expect_disk_usage = 10.4
        update_info = HardwareData(
            timestamp="2000/11/11 11:11:11",
            cpu_temperature=expect_cpu_temperature,
            cpu_usage=expect_cpu_usage,
            memory_usage=expect_memory_usage,
            disk_usage=expect_disk_usage,
        )

        hwmonitor.update_data(data=update_info)

        assert hwmonitor.data is not None
        assert expect_cpu_temperature == hwmonitor.data.cpu_temperature
        assert expect_cpu_usage == hwmonitor.data.cpu_usage
        assert expect_memory_usage == hwmonitor.data.memory_usage
        assert expect_disk_usage == hwmonitor.data.disk_usage

    def test_update_data_to_none(self):
        hwmonitor = HardwareMonitor(
            logger_id="0",
            data=HardwareData(
                timestamp="2000/11/11 11:11:11",
                cpu_temperature=99.9,
                cpu_usage=99.9,
                memory_usage=99.9,
                disk_usage=99.9,
            ),
        )
        update_info = None

        hwmonitor.update_data(data=update_info)

        assert hwmonitor.data is None

    def test_update_error(self):
        hwmonitor = HardwareMonitor(logger_id="0")
        expect_error_level: ErrorType = "CRITICAL"
        expect_error_message = "critical error"
        update_info = HardwareError(
            timestamp="2000/11/11 11:11:11",
            level=expect_error_level,
            message=expect_error_message,
        )

        hwmonitor.update_error(error=update_info)

        assert 1 == len(hwmonitor.error)
        assert expect_error_level == hwmonitor.error[0].level
        assert expect_error_message == hwmonitor.error[0].message

    def test_update_error_to_none(self):
        hwmonitor = HardwareMonitor(
            logger_id="0",
            error=deque[HardwareError](
                [
                    HardwareError(
                        timestamp="2000/11/11 11:11:11",
                        level="CRITICAL",
                        message="critical error",
                    )
                ],
                maxlen=_MAX_MESSAGE_COUNT,
            ),
        )
        update_info = None

        hwmonitor.update_error(error=update_info)

        assert 0 == len(hwmonitor.error)

    def test_update_error_over_limit(self):
        error0_message = "error0_message"
        error1_message = "error1_message"
        error2_message = "error2_message"

        error0 = HardwareError(
            timestamp="2000/11/11 00:00:00", level="CRITICAL", message=error0_message
        )
        error1 = HardwareError(
            timestamp="2000/11/11 10:00:00", level="CRITICAL", message=error1_message
        )
        error2 = HardwareError(
            timestamp="2000/11/11 20:00:00", level="CRITICAL", message=error2_message
        )

        hwmonitor = HardwareMonitor(
            logger_id="0",
            error=deque[HardwareError](
                [error0, error1, error2], maxlen=_MAX_MESSAGE_COUNT
            ),
        )

        over_error_message = "errorover_message"
        over_error = HardwareError(
            timestamp="2000/11/11 23:00:00",
            level="CRITICAL",
            message=over_error_message,
        )

        update_info = over_error

        hwmonitor.update_error(error=update_info)

        assert _MAX_MESSAGE_COUNT == len(hwmonitor.error)
        # 古いエラーが押し出される
        # The oldest error is pushed out
        assert error1 == hwmonitor.error[0]
        # 新しいエラーが末尾
        # The newest error is at the end
        assert over_error == hwmonitor.error[-1]

    def test_is_empty(self):
        hwmonitor = HardwareMonitor(logger_id="0")
        expect = True

        actual = hwmonitor.is_empty()

        assert expect == actual

    def test_is_empty_exist_status(self):
        hwmonitor = HardwareMonitor(
            logger_id="0",
            status=HardwareStatus(timestamp="2000/11/11 11:11:11", status="START"),
        )

        expect = False

        actual = hwmonitor.is_empty()

        assert expect == actual
