# Copyright 2025, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

from collections import deque
from typing import Literal

from raspi_web.const import _MAX_MESSAGE_COUNT
from raspi_web.data.error_type import ErrorType
from raspi_web.data.sensor import Sensor, SensorError, SensorMessage, SensorStatus


class TestSensor:

    def test_update_status(self):
        sensor = Sensor(logger_id="0", model="A352", serial="0001")
        expect: Literal["OK", "NG"] = "OK"
        update_info = SensorStatus(timestamp="2000/11/11 11:11:11", status=expect)

        sensor.update_status(status=update_info)

        assert sensor.status is not None
        assert expect == sensor.status.status

    def test_update_status_to_none(self):
        sensor = Sensor(
            logger_id="0",
            model="A352",
            serial="0001",
            status=SensorStatus(timestamp="2000/11/11 11:11:11", status="OK"),
        )
        update_info = None

        sensor.update_status(status=update_info)

        assert sensor.status is None

    def test_update_error(self):
        sensor = Sensor(logger_id="0", model="A352", serial="0000")
        expect_error_level: ErrorType = "CRITICAL"
        expect_error_message = "critical error"
        update_info = SensorError(
            timestamp="2000/11/11 11:11:11",
            level=expect_error_level,
            message=expect_error_message,
        )

        sensor.update_error(error=update_info)

        assert 1 == len(sensor.error)
        assert expect_error_level == sensor.error[0].level
        assert expect_error_message == sensor.error[0].message

    def test_update_error_to_none(self):
        sensor = Sensor(
            logger_id="0",
            model="A352",
            serial="0000",
            error=deque[SensorError](
                [
                    SensorError(
                        timestamp="2000/11/11 11:11:11",
                        level="CRITICAL",
                        message="critical error",
                    )
                ],
                maxlen=_MAX_MESSAGE_COUNT,
            ),
        )
        update_info = None

        sensor.update_error(error=update_info)

        assert 0 == len(sensor.error)

    def test_update_error_over_limit(self):
        error0_message = "error0_message"
        error1_message = "error1_message"
        error2_message = "error2_message"

        error0 = SensorError(
            timestamp="2000/11/11 00:00:00", level="CRITICAL", message=error0_message
        )
        error1 = SensorError(
            timestamp="2000/11/11 10:00:00", level="CRITICAL", message=error1_message
        )
        error2 = SensorError(
            timestamp="2000/11/11 20:00:00", level="CRITICAL", message=error2_message
        )

        sensor = Sensor(
            logger_id="0",
            model="A352",
            serial="0000",
            error=deque[SensorError](
                [error0, error1, error2], maxlen=_MAX_MESSAGE_COUNT
            ),
        )

        over_error_message = "errorover_message"
        over_error = SensorError(
            timestamp="2000/11/11 23:00:00",
            level="CRITICAL",
            message=over_error_message,
        )

        update_info = over_error

        sensor.update_error(error=update_info)

        assert _MAX_MESSAGE_COUNT == len(sensor.error)
        # 古いエラーが押し出される
        # The oldest error is pushed out
        assert error1 == sensor.error[0]
        # 新しいエラーが末尾
        # The newest error is at the end
        assert over_error == sensor.error[-1]

    def test_update_loss(self):
        sensor = Sensor(logger_id="0", model="A352", serial="0000")
        expect_loss_message = "ロスしました"
        update_info = SensorMessage(
            timestamp="2000/11/11 11:11:11",
            message=expect_loss_message,
        )

        sensor.update_loss(loss=update_info)

        assert 1 == len(sensor.loss)
        assert expect_loss_message == sensor.loss[0].message

    def test_update_loss_to_none(self):
        sensor = Sensor(
            logger_id="0",
            model="A352",
            serial="0000",
            loss=deque[SensorMessage](
                [
                    SensorMessage(
                        timestamp="2000/11/11 11:11:11",
                        message="loss loss",
                    )
                ],
                maxlen=_MAX_MESSAGE_COUNT,
            ),
        )
        update_info = None

        sensor.update_loss(loss=update_info)

        assert 0 == len(sensor.loss)

    def test_update_loss_over_limit(self):
        loss0_message = "loss0_message"
        loss1_message = "loss1_message"
        loss2_message = "loss2_message"

        loss0 = SensorMessage(timestamp="2000/11/11 00:00:00", message=loss0_message)
        loss1 = SensorMessage(timestamp="2000/11/11 10:00:00", message=loss1_message)
        loss2 = SensorMessage(timestamp="2000/11/11 20:00:00", message=loss2_message)

        sensor = Sensor(
            logger_id="0",
            model="A352",
            serial="0000",
            loss=deque[SensorMessage]([loss0, loss1, loss2], maxlen=_MAX_MESSAGE_COUNT),
        )

        over_loss_message = "loss_over_message"
        over_loss = SensorMessage(
            timestamp="2000/11/11 23:00:00",
            message=over_loss_message,
        )

        update_info = over_loss

        sensor.update_loss(loss=update_info)

        assert _MAX_MESSAGE_COUNT == len(sensor.loss)
        # 古いデータ欠落発生情報が押し出される
        # The oldest data loss information is pushed out
        assert loss1 == sensor.loss[0]
        # 新しいデータ欠落発生情報が末尾
        # The newest data loss information is at the end
        assert over_loss == sensor.loss[-1]

    def test_update_abnormal(self):
        sensor = Sensor(logger_id="0", model="A352", serial="0000")
        expect_abnormal_message = "素子異常を検知しました"
        update_info = SensorMessage(
            timestamp="2000/11/11 11:11:11",
            message=expect_abnormal_message,
        )

        sensor.update_abnormal(abnormal=update_info)

        assert 1 == len(sensor.abnormal)
        assert expect_abnormal_message == sensor.abnormal[0].message

    def test_update_abnormal_to_none(self):
        sensor = Sensor(
            logger_id="0",
            model="A352",
            serial="0000",
            abnormal=deque[SensorMessage](
                [
                    SensorMessage(
                        timestamp="2000/11/11 11:11:11",
                        message="abnormal abnormal",
                    )
                ],
                maxlen=_MAX_MESSAGE_COUNT,
            ),
        )
        update_info = None

        sensor.update_abnormal(abnormal=update_info)

        assert 0 == len(sensor.abnormal)

    def test_update_abnormal_over_limit(self):
        abnormal0_message = "abnormal0_message"
        abnormal1_message = "abnormal1_message"
        abnormal2_message = "abnormal2_message"

        abnormal0 = SensorMessage(
            timestamp="2000/11/11 00:00:00", message=abnormal0_message
        )
        abnormal1 = SensorMessage(
            timestamp="2000/11/11 10:00:00", message=abnormal1_message
        )
        abnormal2 = SensorMessage(
            timestamp="2000/11/11 20:00:00", message=abnormal2_message
        )

        sensor = Sensor(
            logger_id="0",
            model="A352",
            serial="0000",
            abnormal=deque[SensorMessage](
                [abnormal0, abnormal1, abnormal2], maxlen=_MAX_MESSAGE_COUNT
            ),
        )

        over_abnormal_message = "abnormal_over_message"
        over_abnormal = SensorMessage(
            timestamp="2000/11/11 23:00:00",
            message=over_abnormal_message,
        )

        update_info = over_abnormal

        sensor.update_abnormal(abnormal=update_info)

        assert _MAX_MESSAGE_COUNT == len(sensor.abnormal)
        # 古い素子異常発生情報が押し出される
        # The oldest abnormality information is pushed out
        assert abnormal1 == sensor.abnormal[0]
        # 新しい素子異常発生情報が末尾
        # The newest abnormality information is at the end
        assert over_abnormal == sensor.abnormal[-1]

    def test_is_empty(self):
        sensor = Sensor(logger_id="0", model="A352", serial="0001")

        expect = True

        actual = sensor.is_empty()

        assert expect == actual

    def test_is_empty_exist_status(self):
        sensor = Sensor(
            logger_id="0",
            model="A352",
            serial="0001",
            status=SensorStatus(timestamp="2000/11/11 11:11:11", status="OK"),
        )

        expect = False

        actual = sensor.is_empty()

        assert expect == actual

    def test_lt_1(self):
        sensor1 = Sensor(logger_id="0", model="A352", serial="0001")
        sensor2 = Sensor(logger_id="0", model="A352", serial="0002")

        assert sensor1 < sensor2
        assert not sensor2 < sensor1

    def test_lt_2(self):
        sensor1 = Sensor(logger_id="0", model="A342", serial="0001")
        sensor2 = Sensor(logger_id="0", model="A352", serial="0001")

        assert sensor1 < sensor2
        assert not sensor2 < sensor1

    def test_lt_3(self):
        # logger_id は比較の対象外
        # logger_id is not included in the comparison
        sensor1 = Sensor(logger_id="0", model="A352", serial="0001")
        sensor2 = Sensor(logger_id="1", model="A352", serial="0001")

        assert not sensor1 < sensor2
        assert not sensor2 < sensor1

    def test_sorted(self):
        sensors = sorted(
            [
                Sensor(logger_id="0", model="A352", serial="0003"),
                Sensor(logger_id="0", model="A352", serial="0002"),
                Sensor(logger_id="0", model="A352", serial="0001"),
                Sensor(logger_id="0", model="A342", serial="0004"),
            ]
        )
        assert sensors[0].model == "A342" and sensors[0].serial == "0004"
        assert sensors[1].model == "A352" and sensors[1].serial == "0001"
        assert sensors[2].model == "A352" and sensors[2].serial == "0002"
        assert sensors[3].model == "A352" and sensors[3].serial == "0003"
