# Copyright 2025, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

import copy
import threading
import time
from collections import deque
from typing import Callable, Self

import pytest

from raspi_web.const import _MAX_MESSAGE_COUNT
from raspi_web.data.hwmonitor import (
    HardwareData,
    HardwareError,
    HardwareMonitor,
    HardwareStatus,
)
from raspi_web.data.logger import (
    Logger,
    LoggerError,
    LoggerStatus,
)
from raspi_web.data.sensor import Sensor, SensorError, SensorMessage, SensorStatus
from raspi_web.data.sensor_key import to_sensor_key
from raspi_web.data.summary import SensorSummary
from raspi_web.data.vc_data import FFTData, VCData, VCLevel, VCLevelType
from raspi_web.infra.data_manager import (
    DataManager,
    DataSet,
    _DataSetEntityManager,
    _SummaryEntityManager,
    _VCDataEntityManager,
)
from tests.testutils.data_sample import get_fft_data_sample


@pytest.fixture()
def reset_data_manager():
    # setup
    dm = DataManager.get_instance()
    dm._reset()

    yield

    # teardown
    dm._reset()


class Test_DataSetEntityManager:

    ## DataSet frame
    def test_create_data_set_frame(self):
        expect_logger_id = "hoge"
        manager = _DataSetEntityManager()
        manager._create_data_set_frame(logger_id=expect_logger_id)

        assert 1 == len(manager._data_sets)
        assert expect_logger_id in manager._data_sets

    def test_create_data_set_frame_already_exist(self):
        expect_logger_id = "hoge"
        manager = _DataSetEntityManager()
        manager._data_sets[expect_logger_id] = DataSet()
        manager._create_data_set_frame(logger_id=expect_logger_id)

        assert 1 == len(manager._data_sets)

    def test_remove_data_set_frame(self):
        logger_id = "logger_sample"
        manager = _DataSetEntityManager()
        manager._data_sets[logger_id] = DataSet()

        manager._remove_data_set_frame(logger_id=logger_id)

        assert 0 == len(manager._data_sets)

    def test_remove_data_set_frame_not_empty(self):
        logger_id = "logger_sample"
        manager = _DataSetEntityManager()
        manager._data_sets[logger_id] = DataSet(
            logger=Logger(
                logger_id=logger_id,
                status=LoggerStatus(timestamp="2024/12/12 00:00:00", status="START"),
            )
        )

        manager._remove_data_set_frame(logger_id=logger_id)

        assert 1 == len(manager._data_sets)

    ## Logger handler
    def test_prep_logger_handler(self):
        logger_id = "0"
        topic = f"logger/{logger_id}"

        manager = _DataSetEntityManager()
        result_logger_id, result_data_set, result_logger = manager._prep_logger_handler(
            logger_topic=topic
        )

        assert logger_id == result_logger_id
        assert result_data_set is not None
        assert result_logger is not None

    def test_handle_logger_status(self):
        logger_id = "0"
        expect_status = "START"
        topic = f"logger/{logger_id}"
        message_dict = {"timestamp": "yyyy/mm/dd hh:mm:ss", "status": expect_status}

        manager = _DataSetEntityManager()
        manager.handle_logger_status(topic=topic, message_dict=message_dict)

        data_set = manager._data_sets[logger_id]

        assert expect_status == data_set.logger.status.status  # type: ignore

    def test_handle_logger_status_none(self):
        logger_id = "0"
        topic = f"logger/{logger_id}"
        message_dict: dict = {}

        manager = _DataSetEntityManager()

        logger = Logger(
            logger_id=logger_id,
            status=LoggerStatus(timestamp="2000/01/01 01:01:01", status="START"),
            error=deque[LoggerError](
                [
                    LoggerError(
                        timestamp="2024/11/21 16:30:00",
                        level="CRITICAL",
                        message="critical error",
                    )
                ],
                maxlen=_MAX_MESSAGE_COUNT,
            ),
        )
        manager._data_sets[logger_id] = DataSet(logger=logger)
        manager.handle_logger_status(topic=topic, message_dict=message_dict)

        data_set = manager._data_sets[logger_id]

        assert data_set.logger is not None
        assert data_set.logger.status is None

    def test_handle_logger_status_none_remove_data_set_frame(self):
        logger_id = "0"
        topic = f"logger/{logger_id}"
        message_dict: dict = {}

        manager = _DataSetEntityManager()

        logger = Logger(
            logger_id=logger_id,
            status=LoggerStatus(timestamp="2000/01/01 01:01:01", status="START"),
        )
        manager._data_sets[logger_id] = DataSet(logger=logger)
        manager.handle_logger_status(topic=topic, message_dict=message_dict)

        result = manager._data_sets.get(logger_id)

        assert result is None

    ## Logger Error handler
    def test_handle_logger_error(self):
        logger_id = "0"
        expect_level = "CRITICAL"
        expect_message = "error01"
        topic = f"logger/{logger_id}/error"
        message_dict = {
            "timestamp": "yyyy/mm/dd hh:mm:ss",
            "level": expect_level,
            "message": expect_message,
        }

        manager = _DataSetEntityManager()
        manager.handle_logger_error(topic=topic, message_dict=message_dict)

        data_set = manager._data_sets[logger_id]

        assert expect_level == data_set.logger.error[0].level  # type: ignore
        assert expect_message == data_set.logger.error[0].message  # type: ignore

    def test_handle_logger_error_none(self):
        logger_id = "0"
        topic = f"logger/{logger_id}/error"
        message_dict: dict = {}

        manager = _DataSetEntityManager()

        logger = Logger(
            logger_id=logger_id,
            status=LoggerStatus(timestamp="2000/01/01 01:01:01", status="START"),
            error=deque[LoggerError](
                [
                    LoggerError(
                        timestamp="2024/11/21 16:30:00",
                        level="CRITICAL",
                        message="critical error",
                    )
                ],
                maxlen=_MAX_MESSAGE_COUNT,
            ),
        )
        manager._data_sets[logger_id] = DataSet(logger=logger)
        manager.handle_logger_error(topic=topic, message_dict=message_dict)

        data_set = manager._data_sets[logger_id]

        assert data_set.logger is not None
        assert len(data_set.logger.error) == 0

    def test_handle_logger_error_none_remove_data_set_frame(self):
        logger_id = "0"
        topic = f"logger/{logger_id}/error"
        message_dict: dict = {}

        manager = _DataSetEntityManager()

        logger = Logger(
            logger_id=logger_id,
            error=deque[LoggerError](
                [
                    LoggerError(
                        timestamp="2024/11/21 16:30:00",
                        level="CRITICAL",
                        message="critical error",
                    )
                ],
                maxlen=_MAX_MESSAGE_COUNT,
            ),
        )
        manager._data_sets[logger_id] = DataSet(logger=logger)
        manager.handle_logger_error(topic=topic, message_dict=message_dict)

        result = manager._data_sets.get(logger_id)

        assert result is None

    ## Sensor handler
    def test_prep_sensor_handler(self):
        logger_id = "0"
        model = "A352"
        serial = "0001"

        topic = f"logger/{logger_id}/sensor/{model}/{serial}"
        manager = _DataSetEntityManager()
        (
            result_logger_id,
            result_model,
            result_serial,
            result_data_set,
            result_sensor,
        ) = manager._prep_sensor_handler(sensor_topic=topic)

        assert logger_id == result_logger_id
        assert model == result_model
        assert serial == result_serial
        assert result_data_set is not None
        assert result_sensor is not None

    def test_handle_sensor_status(self):
        logger_id = "0"
        model = "A352"
        serial = "0001"
        expect_status = "OK"

        topic = f"logger/{logger_id}/sensor/{model}/{serial}"
        message_dict = {"timestamp": "yyyy/mm/dd hh:mm:ss", "status": expect_status}

        manager = _DataSetEntityManager()
        manager.handle_sensor_status(topic=topic, message_dict=message_dict)

        data_set = manager._data_sets[logger_id]

        assert len(data_set.sensors) == 1
        assert expect_status == data_set.sensors[0].status.status  # type: ignore

    def test_handle_sensor_status_none(self):
        logger_id = "0"
        model = "A352"
        serial = "0001"

        topic = f"logger/{logger_id}/sensor/{model}/{serial}"
        message_dict: dict = {}

        sensor = Sensor(
            logger_id=logger_id,
            model=model,
            serial=serial,
            status=SensorStatus(timestamp="2000/01/01 00:00:00", status="OK"),
            error=deque[SensorError](
                [
                    SensorError(
                        timestamp="2000/01/01 00:00:00",
                        level="CRITICAL",
                        message="critical error",
                    )
                ],
                maxlen=_MAX_MESSAGE_COUNT,
            ),
        )

        manager = _DataSetEntityManager()
        manager._data_sets[logger_id] = DataSet(sensors=[sensor])
        manager.handle_sensor_status(topic=topic, message_dict=message_dict)

        data_set = manager._data_sets[logger_id]

        assert data_set.sensors[0].status is None

    def test_handle_sensor_status_none_remove_data_set_frame(self):
        logger_id = "0"
        model = "A352"
        serial = "0001"

        topic = f"logger/{logger_id}/sensor/{model}/{serial}"
        message_dict: dict = {}

        sensor = Sensor(
            logger_id=logger_id,
            model=model,
            serial=serial,
            status=SensorStatus(timestamp="2000/01/01 00:00:00", status="OK"),
        )

        manager = _DataSetEntityManager()
        manager._data_sets[logger_id] = DataSet(sensors=[sensor])
        manager.handle_sensor_status(topic=topic, message_dict=message_dict)

        result = manager._data_sets.get(logger_id)

        assert result is None

    ## Sensor Error handler
    def test_handle_sensor_error(self):
        logger_id = "0"
        model = "A352"
        serial = "0001"
        expect_level = "CRITICAL"
        expect_message = "error01"

        topic = f"logger/{logger_id}/sensor/{model}/{serial}/error"
        message_dict = {
            "timestamp": "yyyy/mm/dd hh:mm:ss",
            "level": expect_level,
            "message": expect_message,
        }
        manager = _DataSetEntityManager()
        manager.handle_sensor_error(topic=topic, message_dict=message_dict)

        data_set = manager._data_sets[logger_id]

        assert len(data_set.sensors) == 1
        assert expect_level == data_set.sensors[0].error[0].level  # type: ignore
        assert expect_message == data_set.sensors[0].error[0].message  # type: ignore

    def test_handle_sensor_error_none(self):
        logger_id = "0"
        model = "A352"
        serial = "0001"

        topic = f"logger/{logger_id}/sensor/{model}/{serial}/error"
        message_dict: dict = {}

        sensor = Sensor(
            logger_id=logger_id,
            model=model,
            serial=serial,
            status=SensorStatus(timestamp="2000/01/01 00:00:00", status="OK"),
            error=deque[SensorError](
                [
                    SensorError(
                        timestamp="2000/01/01 00:00:00",
                        level="CRITICAL",
                        message="critical error",
                    )
                ],
                maxlen=_MAX_MESSAGE_COUNT,
            ),
        )

        manager = _DataSetEntityManager()
        manager._data_sets[logger_id] = DataSet(sensors=[sensor])
        manager.handle_sensor_error(topic=topic, message_dict=message_dict)

        data_set = manager._data_sets[logger_id]

        assert len(data_set.sensors[0].error) == 0

    def test_handle_sensor_error_none_remove_data_set_frame(self):
        logger_id = "0"
        model = "A352"
        serial = "0001"

        topic = f"logger/{logger_id}/sensor/{model}/{serial}/error"
        message_dict: dict = {}

        sensor = Sensor(
            logger_id=logger_id,
            model=model,
            serial=serial,
            error=deque[SensorError](
                [
                    SensorError(
                        timestamp="2000/01/01 00:00:00",
                        level="CRITICAL",
                        message="critical error",
                    )
                ],
                maxlen=_MAX_MESSAGE_COUNT,
            ),
        )

        manager = _DataSetEntityManager()
        manager._data_sets[logger_id] = DataSet(sensors=[sensor])
        manager.handle_sensor_error(topic=topic, message_dict=message_dict)

        result = manager._data_sets.get(logger_id)

        assert result is None

    ## Sensor Loss handler
    def test_handle_sensor_loss(self):
        logger_id = "0"
        model = "A352"
        serial = "0001"
        expect_message = "loss01"

        topic = f"logger/{logger_id}/sensor/{model}/{serial}/loss"
        message_dict = {
            "timestamp": "yyyy/mm/dd hh:mm:ss",
            "message": expect_message,
        }
        manager = _DataSetEntityManager()
        manager.handle_sensor_loss(topic=topic, message_dict=message_dict)

        data_set = manager._data_sets[logger_id]

        assert len(data_set.sensors) == 1
        assert expect_message == data_set.sensors[0].loss[0].message  # type: ignore

    def test_handle_sensor_loss_none(self):
        logger_id = "0"
        model = "A352"
        serial = "0001"

        topic = f"logger/{logger_id}/sensor/{model}/{serial}/loss"
        message_dict: dict = {}

        sensor = Sensor(
            logger_id=logger_id,
            model=model,
            serial=serial,
            status=SensorStatus(timestamp="2000/01/01 00:00:00", status="OK"),
            loss=deque[SensorMessage](
                [
                    SensorMessage(
                        timestamp="2000/01/01 00:00:00",
                        message="loss01",
                    )
                ],
                maxlen=_MAX_MESSAGE_COUNT,
            ),
        )

        manager = _DataSetEntityManager()
        manager._data_sets[logger_id] = DataSet(sensors=[sensor])
        manager.handle_sensor_loss(topic=topic, message_dict=message_dict)

        data_set = manager._data_sets[logger_id]

        assert len(data_set.sensors[0].loss) == 0

    def test_handle_sensor_loss_none_remove_data_set_frame(self):
        logger_id = "0"
        model = "A352"
        serial = "0001"

        topic = f"logger/{logger_id}/sensor/{model}/{serial}/loss"
        message_dict: dict = {}

        sensor = Sensor(
            logger_id=logger_id,
            model=model,
            serial=serial,
            loss=deque[SensorMessage](
                [
                    SensorMessage(
                        timestamp="2000/01/01 00:00:00",
                        message="loss01",
                    )
                ],
                maxlen=_MAX_MESSAGE_COUNT,
            ),
        )

        manager = _DataSetEntityManager()
        manager._data_sets[logger_id] = DataSet(sensors=[sensor])
        manager.handle_sensor_loss(topic=topic, message_dict=message_dict)

        result = manager._data_sets.get(logger_id)

        assert result is None

    ## Sensor Abnormal handler
    def test_handle_sensor_abnormal(self):
        logger_id = "0"
        model = "A352"
        serial = "0001"
        expect_message = "abnormal01"

        topic = f"logger/{logger_id}/sensor/{model}/{serial}/abnormal"
        message_dict = {
            "timestamp": "yyyy/mm/dd hh:mm:ss",
            "message": expect_message,
        }
        manager = _DataSetEntityManager()
        manager.handle_sensor_abnormal(topic=topic, message_dict=message_dict)

        data_set = manager._data_sets[logger_id]

        assert len(data_set.sensors) == 1
        assert expect_message == data_set.sensors[0].abnormal[0].message  # type: ignore

    def test_handle_sensor_abnormal_none(self):
        logger_id = "0"
        model = "A352"
        serial = "0001"

        topic = f"logger/{logger_id}/sensor/{model}/{serial}/abnormal"
        message_dict: dict = {}

        sensor = Sensor(
            logger_id=logger_id,
            model=model,
            serial=serial,
            status=SensorStatus(timestamp="2000/01/01 00:00:00", status="OK"),
            abnormal=deque[SensorMessage](
                [
                    SensorMessage(
                        timestamp="2000/01/01 00:00:00",
                        message="abnormal01",
                    )
                ],
                maxlen=_MAX_MESSAGE_COUNT,
            ),
        )

        manager = _DataSetEntityManager()
        manager._data_sets[logger_id] = DataSet(sensors=[sensor])
        manager.handle_sensor_abnormal(topic=topic, message_dict=message_dict)

        data_set = manager._data_sets[logger_id]

        assert len(data_set.sensors[0].abnormal) == 0

    def test_handle_sensor_abnormal_none_remove_data_set_frame(self):
        logger_id = "0"
        model = "A352"
        serial = "0001"

        topic = f"logger/{logger_id}/sensor/{model}/{serial}/abnormal"
        message_dict: dict = {}

        sensor = Sensor(
            logger_id=logger_id,
            model=model,
            serial=serial,
            abnormal=deque[SensorMessage](
                [
                    SensorMessage(
                        timestamp="2000/01/01 00:00:00",
                        message="loss01",
                    )
                ],
                maxlen=_MAX_MESSAGE_COUNT,
            ),
        )

        manager = _DataSetEntityManager()
        manager._data_sets[logger_id] = DataSet(sensors=[sensor])
        manager.handle_sensor_abnormal(topic=topic, message_dict=message_dict)

        result = manager._data_sets.get(logger_id)

        assert result is None

    ## Hardware Monitor handler
    def test_prep_hwmonitor_handler(self):
        logger_id = "0"
        topic = f"hwmonitor/{logger_id}"

        manager = _DataSetEntityManager()
        result_logger_id, result_data_set, result_hwmonitor = (
            manager._prep_hwmonitor_handler(hwmonitor_topic=topic)
        )

        assert logger_id == result_logger_id
        assert result_data_set is not None
        assert result_hwmonitor is not None

    def test_handle_hwmonitor_status(self):
        logger_id = "0"
        expect_status = "START"
        topic = f"hwmonitor/{logger_id}"
        message_dict = {"timestamp": "yyyy/mm/dd hh:mm:ss", "status": expect_status}

        manager = _DataSetEntityManager()
        manager.handle_hwmonitor_status(topic=topic, message_dict=message_dict)

        data_set = manager._data_sets[logger_id]

        assert expect_status == data_set.hwmonitor.status.status  # type: ignore

    def test_handle_hwmonitor_status_none(self):
        logger_id = "0"
        topic = f"hwmonitor/{logger_id}"
        message_dict: dict = {}

        hwmonitor = HardwareMonitor(
            logger_id=logger_id,
            status=HardwareStatus(timestamp="2000/01/01 01:01:01", status="START"),
            error=deque[HardwareError](
                [
                    HardwareError(
                        timestamp="2024/11/21 16:30:00",
                        level="CRITICAL",
                        message="critical error",
                    )
                ],
                maxlen=_MAX_MESSAGE_COUNT,
            ),
        )

        manager = _DataSetEntityManager()
        manager._data_sets[logger_id] = DataSet(hwmonitor=hwmonitor)
        manager.handle_hwmonitor_status(topic=topic, message_dict=message_dict)

        data_set = manager._data_sets[logger_id]

        assert data_set.hwmonitor is not None
        assert data_set.hwmonitor.status is None

    def test_handle_hwmonitor_status_none_remove_data_set_frame(self):
        logger_id = "0"
        topic = f"hwmonitor/{logger_id}"
        message_dict: dict = {}

        hwmonitor = HardwareMonitor(
            logger_id=logger_id,
            status=HardwareStatus(timestamp="2000/01/01 01:01:01", status="START"),
        )

        manager = _DataSetEntityManager()
        manager._data_sets[logger_id] = DataSet(hwmonitor=hwmonitor)
        manager.handle_hwmonitor_status(topic=topic, message_dict=message_dict)

        result = manager._data_sets.get(logger_id)

        assert result is None

    ### Hardware Monitor Data handler
    def test_handle_hwmonitor_data(self):
        logger_id = "0"
        expect_cpu_temperature = 99.99
        expect_cpu_usage = 99.9
        expect_memory_usage = 99.9
        expect_disk_usage = 99.9
        topic = f"hwmonitor/{logger_id}/data"
        message_dict = {
            "timestamp": "2000/01/01 00:00:00",
            "cpu_temperature": expect_cpu_temperature,
            "cpu_usage": expect_cpu_usage,
            "memory_usage": expect_memory_usage,
            "disk_usage": expect_disk_usage,
        }

        manager = _DataSetEntityManager()
        manager.handle_hwmonitor_data(topic=topic, message_dict=message_dict)

        data_set = manager._data_sets[logger_id]

        assert expect_cpu_temperature == data_set.hwmonitor.data.cpu_temperature  # type: ignore
        assert expect_cpu_usage == data_set.hwmonitor.data.cpu_usage  # type: ignore
        assert expect_memory_usage == data_set.hwmonitor.data.memory_usage  # type: ignore
        assert expect_disk_usage == data_set.hwmonitor.data.disk_usage  # type: ignore

    def test_handle_hwmonitor_data_none(self):
        logger_id = "0"
        topic = f"hwmonitor/{logger_id}/data"
        message_dict: dict = {}

        hwmonitor = HardwareMonitor(
            logger_id=logger_id,
            status=HardwareStatus(timestamp="2000/01/01 01:01:01", status="START"),
            data=HardwareData(
                timestamp="2000/01/01 00:00:00",
                cpu_temperature=99.99,
                cpu_usage=99.99,
                memory_usage=99.9,
                disk_usage=99.9,
            ),
        )

        manager = _DataSetEntityManager()
        manager._data_sets[logger_id] = DataSet(hwmonitor=hwmonitor)
        manager.handle_hwmonitor_data(topic=topic, message_dict=message_dict)

        data_set = manager._data_sets[logger_id]

        assert data_set.hwmonitor is not None
        assert data_set.hwmonitor.data is None

    def test_handle_hwmonitor_data_none_remove_data_set_frame(self):
        logger_id = "0"
        topic = f"hwmonitor/{logger_id}/data"
        message_dict: dict = {}

        hwmonitor = HardwareMonitor(
            logger_id=logger_id,
            data=HardwareData(
                timestamp="2000/01/01 00:00:00",
                cpu_temperature=99.99,
                cpu_usage=99.99,
                memory_usage=99.9,
                disk_usage=99.9,
            ),
        )

        manager = _DataSetEntityManager()
        manager._data_sets[logger_id] = DataSet(hwmonitor=hwmonitor)
        manager.handle_hwmonitor_data(topic=topic, message_dict=message_dict)

        result = manager._data_sets.get(logger_id)

        assert result is None

    ### Hardware Monitor Error handler
    def test_handle_hwmonitor_error(self):
        logger_id = "0"
        expect_level = "CRITICAL"
        expect_message = "error01"
        topic = f"hwmonitor/{logger_id}/error"
        message_dict = {
            "timestamp": "yyyy/mm/dd hh:mm:ss",
            "level": expect_level,
            "message": expect_message,
        }

        manager = _DataSetEntityManager()
        manager.handle_hwmonitor_error(topic=topic, message_dict=message_dict)

        data_set = manager._data_sets[logger_id]

        assert expect_level == data_set.hwmonitor.error[0].level  # type: ignore
        assert expect_message == data_set.hwmonitor.error[0].message  # type: ignore

    def test_handle_hwmonitor_error_none(self):
        logger_id = "0"
        topic = f"hwmonitor/{logger_id}/error"
        message_dict: dict = {}

        hwmonitor = HardwareMonitor(
            logger_id=logger_id,
            status=HardwareStatus(timestamp="2000/01/01 01:01:01", status="START"),
            error=deque[HardwareError](
                [
                    HardwareError(
                        timestamp="2024/11/21 16:30:00",
                        level="CRITICAL",
                        message="critical error",
                    )
                ],
                maxlen=_MAX_MESSAGE_COUNT,
            ),
        )
        manager = _DataSetEntityManager()
        manager._data_sets[logger_id] = DataSet(hwmonitor=hwmonitor)
        manager.handle_hwmonitor_error(topic=topic, message_dict=message_dict)

        data_set = manager._data_sets[logger_id]

        assert data_set.hwmonitor is not None
        assert len(data_set.hwmonitor.error) == 0

    def test_handle_hwmonitor_error_none_remove_data_set_frame(self):
        logger_id = "0"
        topic = f"hwmonitor/{logger_id}/error"
        message_dict: dict = {}

        hwmonitor = HardwareMonitor(
            logger_id=logger_id,
            error=deque[HardwareError](
                [
                    HardwareError(
                        timestamp="2024/11/21 16:30:00",
                        level="CRITICAL",
                        message="critical error",
                    )
                ],
                maxlen=_MAX_MESSAGE_COUNT,
            ),
        )

        manager = _DataSetEntityManager()
        manager._data_sets[logger_id] = DataSet(hwmonitor=hwmonitor)
        manager.handle_hwmonitor_error(topic=topic, message_dict=message_dict)

        result = manager._data_sets.get(logger_id)

        assert result is None


class Test_VCDataEntityManager:

    ## VCData frame
    def test_create_frame(self):
        logger_id = "hoge"
        model = "A352"
        serial = "0000"
        expect_vc_data_key = to_sensor_key(
            logger_id=logger_id, model=model, serial=serial
        )

        manager = _VCDataEntityManager()
        manager._create_frame(vc_data_key=expect_vc_data_key)

        assert 1 == len(manager._vc_data)
        assert expect_vc_data_key in manager._vc_data

    def test_create_frame_already_exist(self):
        logger_id = "hoge"
        model = "A352"
        serial = "0000"

        manager = _VCDataEntityManager()
        vc_data_key = to_sensor_key(logger_id=logger_id, model=model, serial=serial)
        manager._vc_data[vc_data_key] = VCData()
        manager._create_frame(vc_data_key=vc_data_key)

        assert 1 == len(manager._vc_data)

    def test_remove_frame(self):
        logger_id = "logger_sample"
        model = "A352"
        serial = "0000"

        manager = _VCDataEntityManager()
        vc_data_key = to_sensor_key(logger_id=logger_id, model=model, serial=serial)
        manager._vc_data[vc_data_key] = VCData()

        manager._remove_frame(vc_data_key=vc_data_key)

        assert 0 == len(manager._vc_data)

    def test_remove_frame_not_empty(self):
        logger_id = "logger_sample"
        model = "A352"
        serial = "0000"

        manager = _VCDataEntityManager()
        vc_data_key = to_sensor_key(logger_id=logger_id, model=model, serial=serial)
        manager._vc_data[vc_data_key] = VCData(
            vc_level=VCLevel(timestamp="2024/12/12 00:00:00", level="OA")
        )

        manager._remove_frame(vc_data_key=vc_data_key)

        assert 1 == len(manager._vc_data)

    ## VCData handler
    def test_prep_handler(self):
        logger_id = "0"
        model = "A352"
        serial = "0001"

        vc_data_key = to_sensor_key(logger_id=logger_id, model=model, serial=serial)

        topic = f"logger/{logger_id}/sensor/{model}/{serial}/vc"

        manager = _VCDataEntityManager()
        result_key, result_vc_data = manager._prep_handler(vc_data_topic=topic)

        assert vc_data_key == result_key
        assert result_vc_data is not None

    ## VCData Level handler
    def test_handle_level(self):
        logger_id = "0"
        model = "A352"
        serial = "0001"
        expect_vc_level = "OA"

        topic = f"logger/{logger_id}/sensor/{model}/{serial}/vc"
        message_dict = {"timestamp": "2020/01/01 00:00:00", "level": expect_vc_level}

        manager = _VCDataEntityManager()
        manager.handle_level(topic=topic, message_dict=message_dict)

        vc_data_key = to_sensor_key(logger_id=logger_id, model=model, serial=serial)
        vc_data = manager._vc_data[vc_data_key]

        assert expect_vc_level == vc_data.vc_level.level  # type: ignore

    def test_handle_level_none(self):
        logger_id = "0"
        model = "A352"
        serial = "0001"

        topic = f"logger/{logger_id}/sensor/{model}/{serial}/vc"
        message_dict: dict = {}

        vc_level = VCLevel(timestamp="2000/01/01 00:00:00", level="A")
        fft_data = FFTData(timestamp="2000/01/01 00:00:00", value=get_fft_data_sample())

        manager = _VCDataEntityManager()
        vc_data_key = to_sensor_key(logger_id=logger_id, model=model, serial=serial)
        manager._vc_data[vc_data_key] = VCData(vc_level=vc_level, fft_data=fft_data)
        manager.handle_level(topic=topic, message_dict=message_dict)

        vc_data = manager._vc_data[vc_data_key]

        assert vc_data.vc_level is None

    def test_handle_level_none_remove_frame(self):
        logger_id = "0"
        model = "A352"
        serial = "0001"

        topic = f"logger/{logger_id}/sensor/{model}/{serial}/vc"
        message_dict: dict = {}

        vc_level = VCLevel(timestamp="2000/01/01 00:00:00", level="A")

        manager = _VCDataEntityManager()
        vc_data_key = to_sensor_key(logger_id=logger_id, model=model, serial=serial)
        manager._vc_data[vc_data_key] = VCData(vc_level=vc_level)
        manager.handle_level(topic=topic, message_dict=message_dict)

        result = manager._vc_data.get(vc_data_key)

        assert result is None

    ## VCData FFT handler
    def test_handle_fft(self):
        logger_id = "0"
        model = "A352"
        serial = "0001"
        expect_fft = get_fft_data_sample()

        topic = f"logger/{logger_id}/sensor/{model}/{serial}/fft"
        message_dict = {"timestamp": "2020/01/01 00:00:00", "value": expect_fft}

        manager = _VCDataEntityManager()
        manager.handle_fft(topic=topic, message_dict=message_dict)

        vc_data_key = to_sensor_key(logger_id=logger_id, model=model, serial=serial)
        vc_data = manager._vc_data[vc_data_key]

        assert expect_fft == vc_data.fft_data.value  # type: ignore

    def test_handle_fft_none(self):
        logger_id = "0"
        model = "A352"
        serial = "0001"

        topic = f"logger/{logger_id}/sensor/{model}/{serial}/fft"
        message_dict: dict = {}

        vc_level = VCLevel(timestamp="2000/01/01 00:00:00", level="A")
        fft_data = FFTData(timestamp="2000/01/01 00:00:00", value=get_fft_data_sample())

        manager = _VCDataEntityManager()
        vc_data_key = to_sensor_key(logger_id=logger_id, model=model, serial=serial)
        manager._vc_data[vc_data_key] = VCData(vc_level=vc_level, fft_data=fft_data)
        manager.handle_fft(topic=topic, message_dict=message_dict)

        vc_data = manager._vc_data[vc_data_key]

        assert vc_data.fft_data is None

    def test_handle_fft_none_remove_frame(self):
        logger_id = "0"
        model = "A352"
        serial = "0001"

        topic = f"logger/{logger_id}/sensor/{model}/{serial}/fft"
        message_dict: dict = {}

        fft_data = FFTData(timestamp="2000/01/01 00:00:00", value=get_fft_data_sample())

        manager = _VCDataEntityManager()
        vc_data_key = to_sensor_key(logger_id=logger_id, model=model, serial=serial)
        manager._vc_data[vc_data_key] = VCData(fft_data=fft_data)
        manager.handle_fft(topic=topic, message_dict=message_dict)

        result = manager._vc_data.get(vc_data_key)

        assert result is None


class Test_SummaryEntityManager:
    def test_create_frame(self):
        logger_id = "hoge"
        model = "A352"
        serial = "0000"
        expect_summary_key = to_sensor_key(
            logger_id=logger_id, model=model, serial=serial
        )
        manager = _SummaryEntityManager()
        manager._create_frame(summary_key=expect_summary_key)

        assert 1 == len(manager._summary)
        assert expect_summary_key in manager._summary

    def test_create_frame_already_exist(self):
        logger_id = "hoge"
        model = "A352"
        serial = "0000"
        manager = _SummaryEntityManager()
        summary_key = to_sensor_key(logger_id=logger_id, model=model, serial=serial)

        manager._summary[summary_key] = SensorSummary()
        manager._create_frame(summary_key=summary_key)

        assert 1 == len(manager._summary)

    def test_remove_frame(self):
        logger_id = "logger_sample"
        model = "A352"
        serial = "0000"
        summary_key = to_sensor_key(logger_id=logger_id, model=model, serial=serial)

        manager = _SummaryEntityManager()
        manager._summary[summary_key] = SensorSummary()
        manager._alert[summary_key] = ""

        manager._remove_frame(summary_key=summary_key)

        assert 0 == len(manager._summary)
        assert 0 == len(manager._alert)

    def test_remove_frame_not_empty(self):
        logger_id = "logger_sample"
        model = "A352"
        serial = "0000"
        summary_key = to_sensor_key(logger_id=logger_id, model=model, serial=serial)

        manager = _SummaryEntityManager()
        manager._summary[summary_key] = SensorSummary(summary={"K": "V"})
        manager._alert[summary_key] = ""

        manager._remove_frame(summary_key=summary_key)

        assert 1 == len(manager._summary)
        assert 1 == len(manager._alert)

    def test_prep_handler(self):
        logger_id = "0"
        model = "A352"
        serial = "0001"
        summary_key = to_sensor_key(logger_id=logger_id, model=model, serial=serial)
        topic = f"summary/{logger_id}/{model}/{serial}"

        manager = _SummaryEntityManager()
        result_key, result_summary = manager._prep_handler(summary_topic=topic)

        assert summary_key == result_key
        assert result_summary is not None

    def test_handle_summary_dict(self):
        logger_id = "0"
        model = "A352"
        serial = "0001"
        cont = {"K": "V", "alert": ""}

        topic = f"summary/{logger_id}/{model}/{serial}"
        message_dict = {"timestamp": "2020/01/01 00:00:00"}
        message_dict.update(cont)

        manager = _SummaryEntityManager()
        manager.handle_summary(topic=topic, message_dict=message_dict)

        summary_key = to_sensor_key(logger_id=logger_id, model=model, serial=serial)
        summary = manager._summary[summary_key]
        assert "V" == summary.summary.get("K")  # type: ignore

        assert "" == manager._alert[summary_key]

    def test_handle_summary_dict_none(self):
        logger_id = "0"
        model = "A352"
        serial = "0001"

        prep_dict = {"timestamp": "2020/01/01 00:00:00"}
        prep_dict.update({"K": "V"})

        summary_key = to_sensor_key(logger_id=logger_id, model=model, serial=serial)
        manager = _SummaryEntityManager()
        manager._summary[summary_key] = SensorSummary(prep_dict)
        manager._alert[summary_key] = ""

        topic = f"summary/{logger_id}/{model}/{serial}"
        message_dict: dict = {}
        manager.handle_summary(topic=topic, message_dict=message_dict)

        assert summary_key not in manager._summary
        assert summary_key not in manager._alert


class TestDataManager:

    ## Queries
    def test_get_data_sets(self, reset_data_manager):
        logger0_id = "0"
        logger1_id = "1"

        expect_logger0 = Logger(
            logger_id=logger0_id,
            status=LoggerStatus(timestamp="2024/12/12 12:12:12", status="START"),
            error=deque(
                [
                    LoggerError(
                        timestamp="2024/12/12 12:12:12",
                        level="CRITICAL",
                        message="hoge",
                    )
                ],
                maxlen=_MAX_MESSAGE_COUNT,
            ),
        )

        expect_sensor0 = Sensor(
            logger_id=logger0_id,
            model="A352",
            serial="0000",
            status=SensorStatus(timestamp="2024/12/01 00:00:00", status="OK"),
        )
        expect_sensor1 = Sensor(
            logger_id=logger1_id,
            model="A352",
            serial="0001",
            status=SensorStatus(timestamp="2024/12/01 10:00:00", status="NG"),
        )
        dm = DataManager.get_instance()
        dm._dataset._data_sets[logger0_id] = DataSet(
            logger=expect_logger0, sensors=[expect_sensor0]
        )
        dm._dataset._data_sets[logger1_id] = DataSet(sensors=[expect_sensor1])

        actual_data_set = dm.get_data_sets()

        assert 2 == len(actual_data_set)

    def test_get_vc_data(self, reset_data_manager):
        logger_id = "0"
        model = "A352"
        serial = "0001"
        expect_level: VCLevelType = "A"
        expect_fft = get_fft_data_sample()

        dm = DataManager.get_instance()
        vc_data_key = to_sensor_key(logger_id=logger_id, model=model, serial=serial)
        dm._vc_data._vc_data[vc_data_key] = VCData(
            vc_level=VCLevel(timestamp="2000/01/01 00:00:00", level=expect_level),
            fft_data=FFTData(timestamp="2000/01/01 00:00:00", value=expect_fft),
        )
        actual = dm.get_vc_data(logger_id=logger_id, model=model, serial=serial)

        assert expect_level == actual.vc_level.level  # type: ignore
        assert expect_fft == actual.fft_data.value  # type: ignore

    def test_get_vc_data_none(self, reset_data_manager):
        logger_id = "0"
        model = "A352"
        serial = "0001"

        dm = DataManager.get_instance()

        actual = dm.get_vc_data(logger_id=logger_id, model=model, serial=serial)

        assert actual is None


class TestMultiThread:
    """
    マルチスレッドの動作確認

    Check the operation of multithreading
    """

    data: list[int] = []
    lock = threading.Lock()

    def _reset_data(self):
        self.data = []

    def _lock_decorator(func: Callable):  # type: ignore
        def _wrapper(self: Self, *args, **kwargs):
            with self.lock:
                result = func(self, *args, **kwargs)
            return result

        return _wrapper

    @_lock_decorator
    def get_data(self, sleep_time: int, get_result: list[int]):
        time.sleep(sleep_time)
        # threadからデータを取り出せないため引数に渡したlistにつめる
        # Because data cannot be retrieved from the thread, it is put into the list passed as an argument
        get_result.extend(copy.deepcopy(self.data))
        return copy.deepcopy(self.data)

    @_lock_decorator
    def register_data(self, sleep_time: int, data: int):
        time.sleep(sleep_time)
        self.data.append(data)

    def test_simulate_get_loggers_in_register_logger(self):
        """
        ロガーの登録処理中にロガー取得処理が実行される場合をシミュレーション

        前提: dataは空

        インプット:
            - 登録スレッド、取得スレッドを作成、実行し、登録処理にスリープを入れることで
              登録処理実行中に取得処理を行う

        アウトプット:
            - スレッドがロックされているため、取得処理の結果が登録処理後のものになる

        Simulate the case where the logger retrieval process is executed
        during the logger registration process

        Prerequisite: data is empty

        Input:
            - Create and execute registration and retrieval threads, and perform \
              the retrieval process during the registration process by inserting a sleep \
              in the registration process

        Output:
            - Since the thread is locked, the result of the retrieval process will be \
              after the registration process
        """
        self._reset_data()

        # thread内での戻り値を入れる
        # Store the return value in the thread
        get_result: list[int] = []

        data_for_register = 1

        # 登録処理に3秒のスリープ
        # 3 seconds sleep in the registration process
        register_thread = threading.Thread(
            target=self.register_data,
            name="register_data_thread",
            args=[3, data_for_register],
            daemon=True,
        )
        get_thread = threading.Thread(
            target=self.get_data,
            name="get_data_thread",
            args=[0, get_result],
            daemon=True,
        )

        register_thread.start()
        # 必ず登録処理が先に始まるようにする
        # Ensure that the registration process starts first
        time.sleep(1)
        get_thread.start()

        register_thread.join()
        get_thread.join()

        assert data_for_register == get_result[0]

    def test_simulate_register_logger_in_get_loggers(self):
        """
        ロガーの取得処理中にロガー登録処理が実行される場合をシミュレーション

        前提: dataは空

        インプット:
            - 登録スレッド、取得スレッドを作成、実行し、取得処理にスリープを入れることで取得処理実行中に登録処理を行う

        アウトプット:
            - スレッドがロックされているため、取得処理の結果が登録処理前のものになる
            - 処理後にdataを見ると登録処理後の値が入っている

        Simulate the case where the logger registration process is executed during
        the logger retrieval process

        Prerequisite: data is empty

        Input:
            - Create and execute registration and retrieval threads, and perform \
                the registration process during the retrieval process by inserting \
                a sleep in the retrieval process

        Output:
            - Since the thread is locked, the result of the retrieval process will be \
                before the registration process
            - After the process, the data contains the value after the registration process
        """
        self._reset_data()

        # thread内での戻り値を入れる
        # Store the return value in the thread
        get_result: list[int] = []

        data_for_register = 1

        # 取得処理に3秒のスリープ
        # 3 seconds sleep in the retrieval process
        get_thread = threading.Thread(
            target=self.get_data,
            name="get_data_thread",
            args=[3, get_result],
            daemon=True,
        )
        register_thread = threading.Thread(
            target=self.register_data,
            name="register_data_thread",
            args=[0, data_for_register],
            daemon=True,
        )

        get_thread.start()
        # 必ず取得処理が先に始まるようにする
        # Ensure that the retrieval process starts first
        time.sleep(1)
        register_thread.start()

        get_thread.join()
        register_thread.join()

        assert 0 == len(get_result)
        assert data_for_register == self.data[0]
