# Copyright 2025, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

from typing import Callable, ParamSpec, TypeVar

# デコレータの引数にとる関数の引数の型のパラメータ仕様変数
# Parameter specification variable for the argument types of the function taken by the decorator
P = ParamSpec("P")
# デコレータの引数にとる関数の戻り値の型の変数
# Type variable for the return type of the function taken by the decorator
R = TypeVar("R")


class TestDecorator:
    """
    クラス内でのデコレータ動作確認

    Check decorator operation within the class
    """
    _prefix = "pre-"
    _hoge = "hoge"

    @staticmethod
    def _decorator(func: Callable[P, R]) -> Callable[P, R]:
        def _wrapper(*args: P.args, **kwargs: P.kwargs):
            result = func(*args, **kwargs)
            result = args[0]._prefix + result  # type: ignore
            return result

        return _wrapper

    @_decorator
    def _get_hoge(self):
        return self._hoge

    def test_decorator(self):
        expect = self._prefix + self._hoge
        actual = self._get_hoge()

        assert expect == actual

    # decoratorがつけられた関数を変数として渡しても変数を呼び出せばデコレータが呼ばれることを確認
    # Confirm that the decorator is called even if the decorated function is passed as a variable and the variable is called
    sample_function = _get_hoge

    def test_call_handler(self):
        expect = self._prefix + self._hoge
        actual = self.sample_function()

        assert expect == actual
