# Copyright 2025, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

import threading
import time
from typing import Any


class TestLock:
    """
    ロックオブジェクトの違いによる再入可能性の確認

    Verifies reentrancy behavior depending on the type of lock object.
    """

    @staticmethod
    def thread_func(result: list[str], lock: Any) -> None:
        """
        スレッド上で実行する関数

        ロックオブジェクトに対して多段のロックをかけ、脱出できるかテストする

        A function to be executed in a thread.

        Tests whether nested locking on the lock object allows reentrancy.
        """
        # 1段目のロック
        # First-level lock
        with lock:
            # 2段目のロック .. 再入可能でないとここで待ちになる
            # Second-level lock – will block here if not reentrant
            with lock:
                # 2段目成功したので結果を格納して終了
                # If second-level lock succeeds, store result and exit
                result.append("OK!")

    def test_normal_lock(self):
        """
        threading.Lock を使ったテスト

        Test using threading.Lock.
        """
        # ロックオブジェクト
        # Lock object
        lock = threading.Lock()
        # 結果ホルダー
        # Result holder
        result: list[str] = []

        # 処理実行スレッド
        # Thread to execute the function
        thread = threading.Thread(
            target=self.thread_func,
            args=[result, lock],
            # 終了しないため daemon
            # Daemon thread to avoid blocking if it hangs
            daemon=True
        )
        # - 実行
        # - Start
        thread.start()
        # - Wait
        time.sleep(1)

        # 評価
        # Evaluation
        assert len(result) == 0

        # 回収
        # Cleanup
        assert thread.is_alive()
        thread.join(1.0)
        # - まだ終了していない
        # - Still not finished
        assert thread.is_alive()

    def test_reentrant_lock(self):
        """
        threading.RLock を使ったテスト
        Test using threading.RLock.
        """
        # 再入可能ロックオブジェクト
        # Reentrant lock object
        lock = threading.RLock()
        # 結果ホルダー
        # Result holder
        result: list[str] = []

        # 処理実行スレッド
        # Thread to execute the function
        thread = threading.Thread(
            target=self.thread_func,
            args=[result, lock],
            daemon=True  # 終了するけど daemon
        )
        # - 実行
        # - Start
        thread.start()
        # - Wait
        time.sleep(1)

        # 評価
        # Evaluation
        assert len(result) == 1
        assert result[0] == "OK!"

        # 回収
        # Cleanup
        assert not thread.is_alive()
