# Copyright 2025, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

import threading
import time

from raspi_web.util.thread_lock import ThreadLock


class TestThreadLock:
    def test_decorate_only(self):
        """
        デコレータを適用しても単純に関数が呼べること

        Verifies that the function can be called normally even when decorated.
        """

        lock = ThreadLock()

        @lock.decorator
        def func(n: int) -> int:
            count = 0
            for _ in range(n):
                count += 1
            return count

        # 単純にコールできること
        # Ensure the function can be called without issues
        num = func(3)

        assert num == 3

    def test_decorate_with_thread(self):
        """
        スレッドを使ってロックの効果確認

        Verifies the effect of locking when used with threads.
        """

        lock = ThreadLock()
        data = []

        # デコレータ使用
        # Using decorator
        @lock.decorator
        def append_func(num: int) -> None:
            for _ in range(num):
                data.append(num)
                time.sleep(1)

        # Thread #1
        th1 = threading.Thread(target=append_func, args=[3], daemon=True)

        # Thread #2
        th2 = threading.Thread(target=append_func, args=[1], daemon=True)

        # 実行
        # Start threads
        th1.start()
        time.sleep(1)
        th2.start()

        # 終了待ち
        # Wait for threads to finish
        th1.join()
        th2.join()

        # 判定
        # Verify result
        assert data == [3, 3, 3, 1]

    def test_context(self):
        """
        コンテキストマネージャとしても使えるか？

        Checks whether the lock can be used as a context manager.
        """

        lock = ThreadLock()
        data = []

        # コンテキスト使用
        # Using context manager
        def append_func(num: int) -> None:
            with lock:
                for _ in range(num):
                    data.append(num)
                    time.sleep(1)

        # Thread #1
        th1 = threading.Thread(target=append_func, args=[3], daemon=True)

        # Thread #2
        th2 = threading.Thread(target=append_func, args=[1], daemon=True)

        # 実行
        # Start threads
        th1.start()
        time.sleep(1)
        th2.start()

        # 終了待ち
        # Wait for threads to finish
        th1.join()
        th2.join()

        # 判定
        assert data == [3, 3, 3, 1]

    def test_reentrant(self):
        """
        再入可能性の確認

        Verifies reentrancy of the lock.
        """

        lock = ThreadLock()
        data = []

        # ２段階コンテキスト使用
        # Nested context manager usage
        def append_func(num: int) -> None:
            with lock:
                with lock:
                    for _ in range(num):
                        data.append(num)
                        time.sleep(1)

        # Thread #1
        th1 = threading.Thread(target=append_func, args=[3], daemon=True)
        th1.start()

        # 多めに待つ
        # Wait longer to ensure completion
        time.sleep(5)

        # 終了待ち
        # Wait for thread to finish
        th1.join(1.0)

        # デッドロックせずに終了していること
        # Ensure no deadlock occurred and thread finished
        assert not th1.is_alive()
