# Copyright 2025, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

from pathlib import Path
from typing import Any

from raspi_summary.const import ConfigKey, Const
from raspi_summary.domain.summary import SensorConfig, SummaryConfig
from raspi_summary.util import env, logging, message

_logger = logging.get_logger(__name__)


def logger_id() -> str | None:
    """
    設定ファイルの次の項目について、必要なチェックを行い返す：
        LOGGER_ID

    Performs necessary checks for the following config item:
        LOGGER_ID

    Returns:
        str|None: LOGGER_ID
    """
    val = env.get_str(ConfigKey.LOGGER_ID)

    # 必須チェック
    # Required field check
    _defined(ConfigKey.LOGGER_ID, val)

    return val


def output_path() -> str | None:
    """
    設定ファイルの次の項目について、必要なチェックを行い返す：
        OUTPUT_PATH

    Performs necessary checks for the following config item:
        OUTPUT_PATH

    Returns:
        str|None: OUTPUT_PATH
    """
    key = ConfigKey.OUTPUT_PATH
    val = env.get_str(key)

    # 必須チェック & ディレクトリ存在チェック
    # Required field check & directory existence check
    return val if _defined(key, val) and _exist_dir(key, val) else None


def max_measurements() -> int | None:
    """
    設定ファイルの次の項目について、必要なチェックを行い返す：
        MAX_MEASUREMENTS

    Performs necessary checks for the following config item:
        MAX_MEASUREMENTS

    Returns:
        int|None: MAX_MEASUREMENTS
    """
    key = ConfigKey.MAX_MEASUREMENTS
    val = env.get_str(key)

    # 必須チェック & 0以上チェック
    # 必須チェック & 整数チェック & 0以上チェック
    # Required field check, integer validation, and check for value >= 0
    if _defined(key, val) and _integer(key, val) and _greater_than_equal(key, val, 0):
        assert val
        return env.parse_int(val)
    else:
        return None


def sensor_model() -> str | None:
    """
    設定ファイルの次の項目について、必要なチェックを行い返す：
        SENSOR_MODEL

    Performs necessary checks for the following config item:
        SENSOR_MODEL

    Returns:
        str|None: SENSOR_MODEL
    """
    val = env.get_str(ConfigKey.SENSOR_MODEL)

    # 必須チェック
    # Required field check
    _defined(ConfigKey.SENSOR_MODEL, val)

    return val


def summary_config() -> SummaryConfig | None:
    """
    設定ファイルの次の項目について、必要なチェックを行い返す：
        SKIP_SECONDS
        SUMMARY_SECONDS

    Performs necessary checks for the following config item:
        SKIP_SECONDS
        SUMMARY_SECONDS

    Returns:
        SummaryConfig|None: SummaryConfig object
            - None: 設定が不正の場合
                If the configuration is invalid
    """
    # SKIP_SECONDS:
    # 必須チェック & 整数チェック & 0以上チェック
    # Required field check, integer validation, and check for value >= 0
    key = ConfigKey.SKIP_SECONDS
    val = env.get_str(key)
    skip = None
    if _defined(key, val) and _integer(key, val) and _greater_than_equal(key, val, 0):
        assert val
        skip = env.parse_int(val)

    # SUMMARY_SECONDS:
    # 必須チェック & 整数チェック & 1以上チェック
    # Required field check, integer validation, and check for value >= 1
    key = ConfigKey.SUMMARY_SECONDS
    val = env.get_str(key)
    summ = None
    if _defined(key, val) and _integer(key, val) and _greater_than_equal(key, val, 1):
        assert val
        summ = env.parse_int(val)

    if skip is not None and summ is not None:
        return SummaryConfig(skip_seconds=skip, summary_seconds=summ)
    else:
        return None


def message_config() -> message.MessageConfig:
    """
    設定ファイルの次の項目について、必要なチェックを行い返す：
        MESSAGE_HOST
        MESSAGE_PORT
        MESSAGE_CLID

    Performs necessary checks for the following config items:
        MESSAGE_HOST
        MESSAGE_PORT
        MESSAGE_CLID

    Returns:
        message.MessageConfig: MessageConfig object
    """
    # MESSAGE_HOST: 未定義警告のみ
    # MESSAGE_HOST: warning if undefined
    key = ConfigKey.MESSAGE_HOST
    host = env.get_str(key)
    _warn_undefined(key, host)

    # MESSAGE_PORT: 未定義警告 + integer チェック & 0 以上チェック
    # MESSAGE_PORT: warning if undefined + integer check + greater than 0
    key = ConfigKey.MESSAGE_PORT
    val = env.get_str(key)
    _warn_undefined(key, val)
    port = (
        env.parse_int(val)
        if val is not None and _integer(key, val) and _greater_than(key, val, 0)
        else None
    )

    # MESSAGE_CLID: 警告なし
    # MESSAGE_CLID: no warning
    clid = env.get_str(ConfigKey.MESSAGE_CLID)

    return message.MessageConfig(host=host, port=port, clid=clid)


def sensor_config(model: str, serial: str) -> SensorConfig:
    """
    設定ファイルの次の項目について、必要なチェックを行い返す：
        - {MODEL}_{SERIAL}_{AXIS}_TREND_BASE
        - {MODEL}_{SERIAL}_{AXIS}_TREND_ALRM
        - {MODEL}_{SERIAL}_{AXIS}_TREND_TRIP
        - {MODEL}_{SERIAL}_{AXIS}_CHANGE_ALRM
        - {MODEL}_{SERIAL}_{AXIS}_CHANGE_TRIP

    チェック内容：
        - 未定義 (Warning)
        - 数値チェック (Error)

    Performs necessary checks for the following config items:
        - {MODEL}_{SERIAL}_{AXIS}_TREND_BASE
        - {MODEL}_{SERIAL}_{AXIS}_TREND_ALRM
        - {MODEL}_{SERIAL}_{AXIS}_TREND_TRIP
        - {MODEL}_{SERIAL}_{AXIS}_CHANGE_ALRM
        - {MODEL}_{SERIAL}_{AXIS}_CHANGE_TRIP

    Validation includes:
        - Undefined (Warning)
        - Numeric check (Error)

    Returns:
        SummaryConfig: 設定５項目 x XYZC軸 = 20項目を保持した構造体
            SummaryConfig object holding 20 items (5 per axis: XYZC)
            - 各値は None を許容する
                allowing None values
    """

    def _get_monitoring(axis: str, key: str) -> float | None:
        """
        監視項目取得

        Retrieve monitoring value
        """
        fmt = key.format(model=model, serial=serial, axis=axis)
        val = env.get_str(fmt)
        if val is None:
            _warn_undefined(fmt, val)
        elif _numeric(fmt, val):
            return env.parse_float(val)
        return None

    def _get_axis(axis: str) -> SensorConfig.Axis:
        """
        軸別監視項目取得

        Retrieve monitoring values for each axis
        """
        tre_base = _get_monitoring(axis, ConfigKey.PTN_TREND_BASE)
        tre_alrm = _get_monitoring(axis, ConfigKey.PTN_TREND_ALRM)
        tre_trip = _get_monitoring(axis, ConfigKey.PTN_TREND_TRIP)
        chg_alrm = _get_monitoring(axis, ConfigKey.PTN_CHANGE_ALRM)
        chg_trip = _get_monitoring(axis, ConfigKey.PTN_CHANGE_TRIP)
        return SensorConfig.Axis(
            trend_base=tre_base,
            trend_limit_alrm=tre_alrm,
            trend_limit_trip=tre_trip,
            change_limit_alrm=chg_alrm,
            change_limit_trip=chg_trip,
        )

    d = {}
    for axis in Const.AXIS_VALUES:
        d[axis] = _get_axis(axis)

    return SensorConfig(x=d["X"], y=d["Y"], z=d["Z"], c=d["C"])


def _warn_undefined(key: str, val: Any) -> None:
    """
    チェック対象が未定義の場合にログに警告を出力する

    Logs a warning if the target value is undefined

    Args:
        key (str): 対象項目名
            Name of the target item
        val (Any): 対象項目値
            Value of the target item
    """
    if not val:
        _logger.warning(f"{key} is not defined")


def _defined(key: str, val: Any) -> bool:
    """
    チェック対象が存在するか判定する

    存在しない場合はログにエラーを出力する

    Checks whether the target value is defined.

    Logs an error if the value is missing.

    Args:
        key (str): 対象項目名
            Name of the target item
        val (Any): 対象項目値
            Value of the target item

    Returns:
        bool: True=存在する
            True if the value is defined
    """
    res = val is not None
    if not res:
        _logger.error(f"{key} must be defined")
    return res


def _one_of(key: str, val: Any, lst: list[Any]) -> bool:
    """
    チェック対象が規定の値であるかチェックする

    規定の値でない場合はログにエラーを出力する

    Checks whether the target value is one of the allowed values.

    Logs an error if the value is not in the allowed list.

    Args:
        key (str): 対象項目名
            Name of the target item
        val (Any): 対象項目値
            Value of the target item
        lst (list[Any]): 対象項目の規定値リスト
            List of allowed values

    Returns:
        bool: True=規定の値である
            True if the value is allowed
    """
    res = val in lst
    if not res:
        _logger.error(f"{key} must be one of [{', '.join(map(str, lst))}]")
    return res


def _numeric(key: str, val: str | None) -> bool:
    """
    チェック対象が数値化可能な文字列かチェックする

    数値化可能な文字列でない場合はログにエラーを出力する

    Checks whether the target string can be converted to a numeric value.

    Logs an error if the string is not numeric.

    Args:
        key (str): 対象項目名
            Name of the target item
        val (str|None): 対象項目値
            Value of the target item

    Returns:
        bool: True=数値文字列である
            True if the string is numeric
    """
    # ** str.isnumeric() などは「数値の文字のみで構成されるか」をチェックするので使えない
    # ** str.isnumeric() checks whether the string consists only of numeric characters, so it cannot be used here
    res = val is not None and env.parse_float(val) is not None
    if not res:
        _logger.error(f"{key} must be numeric")
    return res


def _integer(key: str, val: str | None) -> bool:
    """
    チェック対象が整数化可能な文字列かチェックする

    整数化可能な文字列でない場合はログにエラーを出力する

    Checks whether the target string can be converted to an integer.

    Logs an error if the string is not an integer.

    Args:
        key (str): 対象項目名
            Name of the target item
        val (str|None): 対象項目値
            Value of the target item

    Returns:
        bool: True=整数化可能な文字列である
            True if the string can be converted to an integer
    """
    # ** str.isnumeric() などは「数値の文字のみで構成されるか」をチェックするので使えない
    # ** str.isnumeric() checks whether the string consists only of numeric characters, so it cannot be used here
    res = val is not None and env.parse_int(val) is not None
    if not res:
        _logger.error(f"{key} must be an integer")
    return res


def _greater_than(key: str, val: str | None, num: int | float) -> bool:
    """
    チェック対象が指定数値より大きいかチェックする

    指定数値より小さい場合はログにエラーを出力する

    Checks whether the target value is greater than the specified number.

    Logs an error if the value is less than or equal to the specified number.

    Args:
        key (str): 対象項目名
            Name of the target item
        val (str|None): 対象項目値
            Value of the target item
        num (int|float): 比較対象値
            Number to compare against

    Returns:
        bool: True=指定数値より大きい
            True if the value is greater than the specified number
    """
    # ここに来る時は必須チェックされている前提
    # Assumes required field check has already been performed
    assert val
    # ここに来る時は数値チェックされている前提
    # Assumes numeric validation has already been performed
    fval = env.parse_float(val)
    assert fval is not None

    # 大小比較
    # Compare values
    res = num < fval
    if not res:
        _logger.error(f"{key} must be greater than {num}")
    return res


def _greater_than_equal(key: str, val: str | None, num: int | float) -> bool:
    """
    チェック対象が指定数値以上かチェックする

    指定数値より小さい場合はログにエラーを出力する

    Checks whether the target value is greater than or equal to the specified number.

    Logs an error if the value is less than the specified number.

    Args:
        key (str): 対象項目名
            Name of the target item
        val (str|None): 対象項目値
            Value of the target item
        num (int|float): 比較対象値
            Number to compare against

    Returns:
        bool: True=指定数値以上
            True if the value is greater than or equal to the specified number
    """
    # ここに来る時は必須チェックされている前提
    # Assumes required field check has already been performed
    assert val
    # ここに来る時は数値チェックされている前提
    # Assumes numeric validation has already been performed
    fval = env.parse_float(val)
    assert fval is not None

    # 大小比較
    # Compare values
    res = num <= fval
    if not res:
        _logger.error(f"{key} must be greater than or equal to {num}")
    return res


def _exist_dir(key: str, val: str | None) -> bool:
    """
    チェック対象が存在するディレクトリであるかチェックする

    ディレクトリが存在しない場合はログにエラーを出力する

    Checks whether the target value is an existing directory.

    Logs an error if the directory does not exist.

    Args:
        key (str): 対象項目名
            Name of the target item
        val (str|None): 対象項目値
            Value of the target item

    Returns:
        bool: True=ディレクトリが存在する
            True if the directory exists
    """
    # ここに来る時は必須チェックされている前提
    # Assumes required field check has already been performed
    assert val

    # 存在チェック & ディレクトリチェック
    # Check existence and whether it's a directory
    path = Path(val)
    res = path.exists() and path.is_dir()
    if not res:
        _logger.error(f"{key} must be an existing directory: {path}")
    return res
