# Copyright 2025, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

import csv
import math
from collections.abc import Generator
from dataclasses import dataclass, field
from pathlib import Path
from typing import Callable, Literal, NamedTuple, TypeAlias

import raspi_summary.util.logging as logging
from raspi_summary.domain.logger import (
    MeasureData,
    MeasureInfo,
    SensorKey,
    _PathElement,
)


class SummaryData(_PathElement):
    """
    ロガーの計測データファイルを集計し、その結果を保存したサマリーファイルを表すクラス

    Class representing a summary file that aggregates logger measurement data files and stores the results
    """

    _KEY_MEASURE = "measure"
    _KEY_MODEL = "model"
    _KEY_SERIAL = "serial"
    _KEY_PHYSICAL = "physical"
    _KEY_X = "X"
    _KEY_Y = "Y"
    _KEY_Z = "Z"
    _KEY_C = "C"
    _KEYS = [
        _KEY_MEASURE,
        _KEY_MODEL,
        _KEY_SERIAL,
        _KEY_PHYSICAL,
        _KEY_X,
        _KEY_Y,
        _KEY_Z,
        _KEY_C,
    ]

    class Value(NamedTuple):
        """
        集計結果の値を保持する構造体

        生成以降個別に更新しないため NamedTuple

        Structure to hold aggregated result values

        NamedTuple is used because values are not updated individually after creation
        """

        x: float
        y: float
        z: float
        c: float

    def __init__(self, path: Path) -> None:
        super().__init__(path)

    def _from_dict(self, d: dict) -> None:
        self.measure = d[self._KEY_MEASURE]
        self.model = d[self._KEY_MODEL]
        self.serial = d[self._KEY_SERIAL]
        self.physical = d[self._KEY_PHYSICAL]
        self.value = self.Value(
            x=d[self._KEY_X], y=d[self._KEY_Y], z=d[self._KEY_Z], c=d[self._KEY_C]
        )

    def _to_dict(self) -> dict[str, str | float]:
        return {
            self._KEY_MEASURE: self.measure,
            self._KEY_MODEL: self.model,
            self._KEY_SERIAL: self.serial,
            self._KEY_PHYSICAL: self.physical,
            self._KEY_X: self.value.x,
            self._KEY_Y: self.value.y,
            self._KEY_Z: self.value.z,
            self._KEY_C: self.value.c,
        }

    @classmethod
    def load(cls, info: MeasureInfo) -> "SummaryData | None":
        """
        MeasureInfo に対応する SummaryData ファイルがあるか検査し、
        ある場合は SummaryData オブジェクトを返す。

        Checks whether a SummaryData file corresponding to the given MeasureInfo exists.
        If it exists, returns a SummaryData object. Otherwise, returns None.

        Args:
            info (MeasureInfo): 計測情報ファイルを表す MeasureInfo オブジェクト
                MeasureInfo object representing the measurement info file

        Returns:
            SummaryData|None:
                - summary.csv ファイルがあれば SummaryData を構築して返す
                    Returns a constructed SummaryData object if summary.csv exists
                - ない場合 None を返す
                    Returns None if the file does not exist
        """
        # info から summary path を解決
        # Resolve summary path from info
        path = cls._create_path(info)

        # - 存在判定
        # - Check existence
        if not path.exists():
            return None

        # SummaryData 化
        # Convert to SummaryData
        data = SummaryData(path)

        # データ読み込み
        # Load data
        with open(path, newline="") as f:
            reader = csv.DictReader(f, quoting=csv.QUOTE_NONNUMERIC)
            for dc in reader:
                data._from_dict(dc)
                return data

        # - データが含まれていない場合
        # - If no data is contained
        raise ValueError("Summary file contains no data")

    @classmethod
    def save(cls, info: MeasureInfo, value: Value) -> "SummaryData":
        """
        MeasureInfo に対応する SummaryData を保存する

        Saves a SummaryData file corresponding to the given MeasureInfo.

        Args:
            info (MeasureInfo): 計測情報ファイルを表す MeasureInfo オブジェクト
                MeasureInfo object representing the measurement info file
            value (SummaryData.Value): 集計結果の値を保持したオブジェクト
                Object holding the aggregated result values

        Returns:
            SummaryData: 保存した情報に対応した SummaryData オブジェクト
                SummaryData object corresponding to the saved information
        """
        # info から summary path を解決
        # Resolve summary path from info
        path = cls._create_path(info)

        # SummaryData 化
        # Convert to SummaryData
        data = SummaryData(path)
        data.measure = info.measure
        data.model = info.key.model
        data.serial = info.key.serial
        data.physical = info.physical
        data.value = value

        # データ保存
        # Save data
        with open(path, mode="w", encoding="UTF-8", newline="\r\n") as f:
            writer = csv.DictWriter(
                f,
                fieldnames=cls._KEYS,
                quoting=csv.QUOTE_NONNUMERIC,
                lineterminator="\n",
            )
            writer.writeheader()
            writer.writerow(data._to_dict())

        return data

    @staticmethod
    def _create_path(info: MeasureInfo) -> Path:
        return Path(info.path.parent, info.path.name.replace("info", "summary"))


class DataSet:
    """
    センサーごとの SummaryData を保持するデータセット

    Dataset that holds SummaryData for each sensor
    """

    class _Entry:
        """
        1 センサーに対応するデータ構造

        Data structure corresponding to a single sensor
        """

        def __init__(self, data: SummaryData) -> None:
            self.datalist: list[SummaryData] = [data]

        def add(self, data: SummaryData) -> None:
            """
            Entry に新しい SummaryData を登録する

            登録するデータの物理量がすでに登録されたものと異なる場合、例外が発生し失敗する

            Registers new SummaryData to the Entry.

            If the physical quantity of the new data differs from the already registered one,
            an exception is raised and the registration fails.

            Args:
                data (SummaryData): 登録するデータ
                    SummaryData to be registered

            Raises:
                ValueError: 登録するデータの物理量がすでに登録されたものと異なる場合
                    If the physical quantity of the new data differs from the existing one
            """
            # 物理量に変化がないか判定する
            # Check if the physical quantity has changed
            init = self.datalist[0]
            if data.physical != init.physical:
                raise ValueError(
                    "Summary data cannot be used "
                    + "because the physical differs from the first: "
                    + f"sensor={data.measure}/{data.model}_{data.serial}, "
                    + f"first={init.physical}, new={data.physical}"
                )

            # 追加
            # Add to the list
            self.datalist.append(data)

    def __init__(self) -> None:
        self._sensors: dict[SensorKey, DataSet._Entry] = {}

    def add(self, key: SensorKey, data: SummaryData) -> None:
        """
        センサーに対する SummaryData を追加する

        そのセンサーについて、登録するデータの物理量がすでに登録されたものと異なる場合、
        ログに WARNING を出力する

        Adds SummaryData for a sensor.

        If the physical quantity of the new data differs from the already registered one
        for that sensor, a WARNING is logged.

        Args:
            key (SensorKey): センサーキー情報
                Sensor key information
            data (SummaryData): 登録するデータ
                SummaryData to be registered
        """
        if key in self._sensors:
            try:
                self._sensors[key].add(data)
            except ValueError as e:
                logging.get_logger(__name__).warning(str(e))
        else:
            self._sensors[key] = DataSet._Entry(data)

    def iterate(self) -> Generator[tuple[SensorKey, list[SummaryData]]]:
        """
        DataSet に保持した SensorKey と SummaryData リストを順次返す

        Iterates over the SensorKey and corresponding list of SummaryData stored in the DataSet.

        Returns:
            Generator[tuple[SensorKey,list[SummaryData]]]:
                SensorKey と センサーごとの SummaryData リストを返す Generator
                Generator yielding SensorKey and list of SummaryData for each sensor
        """
        for k, v in self._sensors.items():
            yield k, v.datalist


AlertType: TypeAlias = Literal["Trip", "Alarm", ""]
"""
サマリー情報の警告状態を示すリテラル型

Literal type indicating the alert status of summary information
"""


@dataclass(init=False)
class SensorSummary:
    """
    センサー単位のサマリー情報を、グラフ要素も含みまとめたクラス

    簡便な使い方をするため init=False

    Class that summarizes sensor-level information including graph elements.

    Uses init=False for simplified usage.
    """

    class Axis(NamedTuple):
        """
        Trend/Change で共用する軸ごとのデータを保持する構造体

        Structure to hold axis-specific data shared by Trend and Change
        """

        value: list[float]
        base: float | None
        limit_alrm: float | None
        limit_trip: float | None

        def to_dict(self, base: bool) -> dict[str, list[float] | float | None]:
            """
            軸データを辞書に変換する

            Converts axis data to a dictionary
            """
            d: dict[str, list[float] | float | None] = {"value": self.value}
            if base:
                d["baseline"] = self.base
            d["limit_alarm"] = self.limit_alrm
            d["limit_trip"] = self.limit_trip
            return d

    datetime: list[str]
    physical: str

    trend_x: Axis
    trend_y: Axis
    trend_z: Axis
    trend_c: Axis

    change_x: Axis
    change_y: Axis
    change_z: Axis
    change_c: Axis

    alert: AlertType

    def to_dict(self) -> dict:
        """
        保持する属性内容を dict 化する

        Converts the stored attributes into a dictionary.

        Returns:
            dict: メッセージ形式に沿った dict
                Dictionary formatted for message output
        """
        return {
            "datetime": self.datetime,
            "physical": self.physical,
            "trend": {
                "X": self.trend_x.to_dict(base=True),
                "Y": self.trend_y.to_dict(base=True),
                "Z": self.trend_z.to_dict(base=True),
                "C": self.trend_c.to_dict(base=True),
            },
            "change": {
                "X": self.change_x.to_dict(base=False),
                "Y": self.change_y.to_dict(base=False),
                "Z": self.change_z.to_dict(base=False),
                "C": self.change_c.to_dict(base=False),
            },
            "alert": self.alert,
        }


@dataclass(init=False)
class SensorAlert:
    """
    SensorSummary に対するアラート情報を抽出したクラス

    Class that extracts alert information from SensorSummary
    """

    datetime: str
    physical: str
    level: AlertType
    trend: list[dict[str, str | float | AlertType | None]]
    change: list[dict[str, str | float | AlertType | None]]

    def __init__(self, summary: SensorSummary):
        # 共通項目を抽出
        # Extract common fields
        self.datetime = summary.datetime[-1]
        self.physical = summary.physical
        # 器を作成
        # Initialize containers
        self.level = ""
        self.trend = []
        self.change = []

    def to_dict(self) -> dict:
        return {
            "datetime": self.datetime,
            "physical": self.physical,
            "level": self.level,
            "trend": self.trend,
            "change": self.change,
        }


@dataclass
class SensorConfig:
    """
    センサーに対するサマリー設定項目を保持するクラス

    Class that holds summary configuration items for a sensor
    """

    class Axis(NamedTuple):
        """
        センサーの軸に対するサマリー設定項目を保持する構造体

        Structure that holds summary configuration items for a sensor axis
        """

        trend_base: float | None = None
        trend_limit_alrm: float | None = None
        trend_limit_trip: float | None = None
        change_limit_alrm: float | None = None
        change_limit_trip: float | None = None

    x: Axis = field(default_factory=Axis)
    y: Axis = field(default_factory=Axis)
    z: Axis = field(default_factory=Axis)
    c: Axis = field(default_factory=Axis)


class SummaryConfig(NamedTuple):
    """
    センサーに共通したサマリー設定項目を保持するクラス

    Class holds common summary configuration for sensors.

    Attributes:
        skip_seconds (int): 読み飛ばす秒数
            Number of seconds to skip
        summary_seconds (int): 集計する秒数
            Number of seconds to summarize
    """

    skip_seconds: int
    summary_seconds: int


class SummaryCalc:
    """
    計測データに対する種々のサマリー計算関数を集めたクラス

    Class that collects various summary calculation functions for measurement data
    """

    def __init__(self, config: SummaryConfig) -> None:
        """
        SummaryCalc コンストラクタ

        SummaryCalc constructor

        Args:
            config (SummaryConfig): サマリー設定オブジェクト
                SummaryConfig object
        """
        self._config = config

    def summarize_measure(
        self, info: MeasureInfo, data: MeasureData
    ) -> SummaryData.Value | None:
        """
        MeasureData ファイルからデータをサマリーする

        MeasureData ファイルはロガーのデータファイルのため、以下の仕様で構成される：
            - 見出し行なしカンマ区切り CSV ファイル
            - サマリー対象の列インデックスは、X=3, Y=4, Z=5

        以下の計算を行う：
            - 先頭から SKIP_SECONDS * SPS 件数のデータは読み飛ばす
            - 以降 SUMMARY_SECONDS * SPS 件数のデータに対し以下を処理する
                - XYZ 三軸を合成し Composite を求め、XYZC を蓄積する
            - 必要なデータ数が集まった場合は XYZC ごとに RMS 計算して返す

        Summarizes data from a MeasureData file.

        The MeasureData file is a logger output with the following format:
            - CSV file without header, comma-separated
            - Target columns for summary: X=3, Y=4, Z=5

        The following calculations are performed:
            - Skip the first SKIP_SECONDS * SPS data points
            - For the next SUMMARY_SECONDS * SPS data points:
                - Combine X, Y, Z axes to compute Composite and hold XYZC
            - If sufficient data is collected, compute RMS for each of XYZC and return

        Args:
            info (MeasureInfo): MeasureInfo オブジェクト
                MeasureInfo object
            data (MeasureData): MeasureData オブジェクト
                MeasureData object

        Returns:
            SummaryData.Value|None: サマリー計算結果の XYZC 数値
                Summary result as XYZC values
                - None: MeasureData のデータ件数が計算データ数に満たない場合
                    None if insufficient data
        """
        # 軸に対する列インデックス
        # Column indices for axes
        xi, yi, zi = 3, 4, 5

        # 読み飛ばしデータ個数
        # Number of data points to skip
        skip = self._config.skip_seconds * info.sps

        # データ個数
        # Number of data points to process
        dmax = self._config.summary_seconds * info.sps

        # ファイル読み込み
        # Read file
        lno, cnt = 0, 0
        xl, yl, zl, cl = [], [], [], []
        with open(data.path, newline="") as f:
            reader = csv.reader(f)
            for row in reader:
                # - スキップ判定
                # - Skip initial data
                lno += 1
                if lno <= skip:
                    continue

                # - 終了判定
                # - Check for end of required data
                cnt += 1
                if cnt > dmax:
                    break

                # - データ格納
                # - Store data
                x, y, z = float(row[xi]), float(row[yi]), float(row[zi])
                xl.append(x)
                yl.append(y)
                zl.append(z)
                cl.append(self.composite(x, y, z))

        # RMS 計算 -- データ数が必要量に満たない場合は集計不可とする
        # RMS calculation -- if not enough data, return None
        if len(xl) >= dmax:
            return SummaryData.Value(
                x=self.rms(xl), y=self.rms(yl), z=self.rms(zl), c=self.rms(cl)
            )
        else:
            return None

    def summarize_sensor(
        self, sums: list[SummaryData], conf: SensorConfig
    ) -> tuple[SensorSummary, SensorAlert]:
        """
        センサーに関する SummaryData をサマリーし、SensorSummary と SensorAlert を返す

        Summarizes SummaryData for a sensor and returns SensorSummary and SensorAlert

        Args:
            sums (list[SummaryData]): センサーに関する SummaryData のリスト
                List of SummaryData for the sensor
            conf (SummaryConfig): 設定ファイルに定義されたセンサーに対する各軸ごとの設定値
                Configuration values for each axis defined in the config file

        Returns:
            tuple[SensorSummary,SensorAlert]: 集計結果の SensorSummary, SensorAlert
                Tuple containing the aggregated SensorSummary and SensorAlert
        """

        def to_axis(
            trend: bool, conf: SensorConfig.Axis, value: list[float]
        ) -> SensorSummary.Axis:
            if trend:
                return SensorSummary.Axis(
                    value=value,
                    base=conf.trend_base,
                    limit_alrm=conf.trend_limit_alrm,
                    limit_trip=conf.trend_limit_trip,
                )
            else:
                return SensorSummary.Axis(
                    value=SummaryCalc.trend_to_change(value),
                    base=None,
                    limit_alrm=conf.change_limit_alrm,
                    limit_trip=conf.change_limit_trip,
                )

        smry = SensorSummary()

        # 固定値
        # Fixed value
        smry.physical = sums[0].physical

        # 集計値
        # Aggregated values
        smry.datetime = [data.measure for data in sums]

        # - Trend
        smry.trend_x = to_axis(trend=True, conf=conf.x, value=[d.value.x for d in sums])
        smry.trend_y = to_axis(trend=True, conf=conf.y, value=[d.value.y for d in sums])
        smry.trend_z = to_axis(trend=True, conf=conf.z, value=[d.value.z for d in sums])
        smry.trend_c = to_axis(trend=True, conf=conf.c, value=[d.value.c for d in sums])

        # - Change
        smry.change_x = to_axis(trend=False, conf=conf.x, value=smry.trend_x.value)
        smry.change_y = to_axis(trend=False, conf=conf.y, value=smry.trend_y.value)
        smry.change_z = to_axis(trend=False, conf=conf.z, value=smry.trend_z.value)
        smry.change_c = to_axis(trend=False, conf=conf.c, value=smry.trend_c.value)

        # アラート計算
        # Alert calculation
        alert = self._alert_summary(smry)

        return smry, alert

    def _alert_summary(self, summary: SensorSummary) -> SensorAlert:
        """
        集計された SensorSummary から、警告メッセージを発する SensorAlert を返す

        Returns a SensorAlert containing warning messages derived from the aggregated SensorSummary.

        Args:
            summary (SensorSummary): 集計された SensorSummary
                Aggregated SensorSummary object

        Returns:
            SensorAlert|None: 警告がある場合 SensorAlert
                SensorAlert if warnings exist
                - None: 警告なし
                    otherwise None
        """
        # 警告スキャナー
        # Alert scanner
        scanner = _AlertScanner(summary)

        # SensorAlert 構築
        # Construct SensorAlert
        alert = SensorAlert(summary)

        # - SensorAlert 構築関数
        # - Function to construct SensorAlert
        def construct_alert(
            trend: bool,
            axis: str,
            data: SensorSummary.Axis,
            level: AlertType,
            value: float,
        ) -> None:
            # Alert データ作成
            # Create alert data
            dd: dict[str, str | float | None] = {
                "axis": axis,
                "type": level,
                "value": value,
            }
            if trend:
                dd["baseline"] = data.base
            dd["limit_alarm"] = data.limit_alrm
            dd["limit_trip"] = data.limit_trip

            # Alert データ追加
            # Add alert data
            if trend:
                alert.trend.append(dd)
            else:
                alert.change.append(dd)

        # - 最新データを対象に警告スキャン, SensorAlert.level 更新
        # - Scan the latest data for alerts, update SensorAlert.level
        alert.level = scanner.scan(slice(-1, -2, -1), construct_alert)

        # SensorSummary 全件スキャン, SensorSummary.alert 更新
        # Scan all SensorSummary entries, update SensorSummary.alert
        # - 処理関数を指定せずレベルのみ調べる
        # - Check only alert levels without specifying a processing function
        summary.alert = scanner.scan(slice(0, len(summary.datetime)), None)

        return alert

    @staticmethod
    def composite(x: float, y: float, z: float) -> float:
        """
        三軸合成計算

        Composite calculation of three axes

        Args:
            x (float): X値
                X value
            y (float): Y値
                Y value
            z (float): Z値
                Z value

        Returns:
            float: 三軸合成値 (ROOT(x**2 + y**2 + z**2))
                Composite value of three axes (√(x² + y² + z²))
        """
        return math.sqrt(x**2 + y**2 + z**2)

    @staticmethod
    def rms(lst: list[float]) -> float:
        """
        RMS 計算

        RMS calculation

        Args:
            lst (list[float]): float 値のリスト
                List of float values

        Returns:
            float: RMS 計算結果
                Result of RMS calculation
        """
        return math.sqrt(sum([v**2 for v in lst]) / len(lst))

    @staticmethod
    def trend_to_change(trend_lst: list[float]) -> list[float]:
        """
        傾向リストから変化量リストを計算

        計算方法： `abs(今回値 - 前回値)`, 初回は 0

        Calculates change values from a trend list.

        Calculation method: `abs(current - previous)`, first value is 0

        Args:
            lst (list[float]): 傾向値のリスト
                List of trend values

        Returns:
            list[float]: 変化量のリスト
                List of change values
        """
        pre: float | None = None
        lst = []
        for val in trend_lst:
            lst.append(0.0) if pre is None else lst.append(abs(val - pre))
            pre = val
        return lst

    @staticmethod
    def over_limit(value_lst: list[float], limit: float | None) -> list[bool] | None:
        """
        閾値を超えているか判定

        Determines whether values exceed the threshold

        Args:
            value_lst (list[float]): 判定する数値リスト
                List of values to check
            limit (float|None): 閾値, None=未定義
                Threshold value; None means undefined

        Returns:
            list[bool]|None: 判定結果のリスト, None=閾値未定義の場合
                List of boolean results, or None if threshold is undefined
        """
        return [v > limit for v in value_lst] if limit is not None else None


class _AlertScanner:
    """
    SensorSummary を対象に警告を走査するクラス

    警告処理関数を指定して走査することで、警告発見時に任意の処理を実行することが可能

    Class for scanning alerts in a SensorSummary

    By specifying a alert handler function, custom actions can be executed when alerts are detected.
    """

    _Axis: TypeAlias = SensorSummary.Axis  # alias

    AlertFunc: TypeAlias = Callable[[bool, str, _Axis, AlertType, float], None]
    """
    警告処理関数の型定義

    Type definition for the alert handler function
    """

    def __init__(self, summary: SensorSummary) -> None:
        self._summary = summary
        self._alerts: list[AlertType] = []

    def scan(self, slice: slice, func: AlertFunc | None) -> AlertType:
        """
        SensorSummary を走査して警告処理関数を実行する関数

        Function to scan SensorSummary and execute the alert handler function

        Args:
            slice (slice): 時系列の軸データから走査する対象範囲を指定する slice
                Slice specifying the target range from time-series axis data
            func (AlertFunc|None): 警告発見時に実行される警告処理関数、Noneの場合は無視される
                Alert handler function executed when a alert is found; ignored if None

        Returns:
            AlertType: 走査した結果、最も高い警告レベルを返す
                Returns the highest alert level found during the scan
        """
        summ = self._summary

        # 過去の警告はクリアする
        # Clear previous alerts
        self._alerts.clear()

        # Trend
        self._scan_graph(
            trend=True,
            laxis=[summ.trend_x, summ.trend_y, summ.trend_z, summ.trend_c],
            slice=slice,
            func=func,
        )

        # Change
        self._scan_graph(
            trend=False,
            laxis=[summ.change_x, summ.change_y, summ.change_z, summ.change_c],
            slice=slice,
            func=func,
        )

        # 最大レベルを返す
        # Return the highest alert level
        return self._max_alert()

    def _scan_graph(
        self, trend: bool, laxis: list[_Axis], slice: slice, func: AlertFunc | None
    ) -> None:
        """
        グラフデータセットごとに警告判定する関数

        Function to evaluate alerts for each graph dataset

        Args:
            trend (bool): 走査している対象が Trend か Change かを示す、True=Trend
                Indicates whether the target being scanned is Trend or Change; True = Trend
            laxis (list[_Axis]): グラフデータセットの XYZC の Axis のリスト
                List of XYZC Axis objects in the graph dataset
            slice (slice): Axis からのデータ走査範囲を指定する slice
                Slice specifying the data scan range from Axis
            func (AlertFunc|None): 警告処理関数
                Alert handler function
        """
        # 軸別に処理
        # Process each axis
        for axis, data in zip("XYZC", laxis):
            self._scan_axis(trend, axis, data, slice, func)

    def _scan_axis(
        self, trend: bool, axis: str, data: _Axis, slice: slice, func: AlertFunc | None
    ) -> None:
        """
        SensorSummary.Axis に対して警告判定をする関数

        Function to evaluate warnings for a SensorSummary.Axis
        """
        # 指定範囲のデータを処理
        # Process data within the specified range
        for value in data.value[slice]:
            # - 警告判定
            # - Evaluate alert level
            level: AlertType = ""
            if data.limit_trip is not None and data.limit_trip <= value:
                level = "Trip"
            elif data.limit_alrm is not None and data.limit_alrm <= value:
                level = "Alarm"

            # - 警告処理
            # - Handle alert
            if level:
                # - レベル登録
                # - Register alert level
                self._alerts.append(level)

                # - 警告関数を実行
                # - Execute alert handler function
                if func is not None:
                    func(trend, axis, data, level, value)

    def _max_alert(self) -> AlertType:
        """
        レベルが大きいアラートを返す

        Returns the highest alert level
        """
        return max(self._alerts) if self._alerts else ""
