# Copyright 2025, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

import logging
import sys
import typing

from raspi_summary.const import ConfigKey, Const

from .env import get_str

Logger: typing.TypeAlias = logging.Logger


def init_logging():
    """
    logging の初期化

    Initialization of logging
    """

    # Log level
    level = get_str(ConfigKey.LOG_LEVEL)
    if level not in Const.LOG_LEVELS:
        level = Const.LOG_LEVEL_DEFAULT

    # Log formatter
    class _Formatter(logging.Formatter):
        def format(self, record: logging.LogRecord) -> str:
            return "%-8s%s %s: %s" % (
                record.levelname,
                self.formatTime(record),
                record.name,
                record.getMessage(),
            )

    # Log handler
    handler = logging.StreamHandler(sys.stdout)
    handler.setFormatter(_Formatter())

    # Init root logger
    root = logging.getLogger()
    root.setLevel(level)
    root.addHandler(handler)


def get_logger(name: str | None) -> Logger:
    """
    logging.getLogger() を呼び出す

    Calls logging.getLogger()
    """
    return logging.getLogger(name)
