# Copyright 2025, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

import csv
import logging
import math
import shutil
from pathlib import Path

from raspi_summary.domain.logger import MeasureData, MeasureInfo, SensorKey
from raspi_summary.domain.summary import (
    DataSet,
    SensorAlert,
    SensorConfig,
    SensorSummary,
    SummaryCalc,
    SummaryConfig,
    SummaryData,
    _AlertScanner,
)
from tests.asset.path import MEASURE_PATH


class TestSummary:

    @staticmethod
    def _create_dummy_summary() -> SummaryData:
        """
        テスト用に SummaryData を構築

        Create a dummy SummaryData object for testing
        """

        data = SummaryData(Path("dummy"))
        data._from_dict(
            {
                "measure": "dummy",
                "model": "A342",
                "serial": "00000100",
                "physical": "Velocity",
                "X": 1.23,
                "Y": 2.34,
                "Z": 3.45,
                "C": 4.56,
            }
        )
        return data

    class TestSummaryData:
        _INFO_CSV = "A342_00000100_info.csv"
        _SUMMARY_CSV = "A342_00000100_summary.csv"
        _INFO_PATH = Path(MEASURE_PATH, "20250801_091011", _INFO_CSV)

        def _copy_info(self, tmpdir: str) -> MeasureInfo:
            # info ファイルを tmpdir にコピー
            # Copy info file to tmpdir
            shutil.copy(self._INFO_PATH, tmpdir)
            info = MeasureInfo(Path(tmpdir, self._INFO_CSV), "dummy")
            assert str(info.path.parent) == tmpdir
            assert info.path.exists()
            return info

        def test_load_without_file(self, tmpdir):
            info = self._copy_info(tmpdir)

            # ファイルがない状態での load()
            # Test load() when summary file does not exist
            data = SummaryData.load(info)

            assert data is None

        def test_load_with_dummy_csv(self, tmpdir):
            info = self._copy_info(tmpdir)

            # ダミーの CSV ファイルを作成
            # Create dummy summary CSV file
            path = Path(tmpdir, self._SUMMARY_CSV)
            fields = [
                "measure",
                "model",
                "serial",
                "physical",
                "X",
                "Y",
                "Z",
                "C",
            ]
            with open(path, mode="w", newline="\r\n") as f:
                writer = csv.DictWriter(
                    f,
                    fieldnames=fields,
                    quoting=csv.QUOTE_NONNUMERIC,
                    lineterminator="\n",
                )
                writer.writeheader()
                writer.writerow(
                    {
                        "measure": "dummy",
                        "model": "A342",
                        "serial": "00000100",
                        "physical": "Velocity",
                        "X": 1.23456,
                        "Y": 2.34567,
                        "Z": 3.45678,
                        "C": 4.56789,
                    }
                )

            assert path.exists()

            # load()
            # Load the summary file
            data = SummaryData.load(info)

            assert data is not None
            assert data.measure == "dummy"
            assert data.model == "A342"
            assert data.serial == "00000100"
            assert data.physical == "Velocity"
            assert data.value.x == 1.23456
            assert data.value.y == 2.34567
            assert data.value.z == 3.45678
            assert data.value.c == 4.56789

        def test_save_with_info(self, tmpdir):
            info = self._copy_info(tmpdir)

            # save() を実行
            # Execute save()
            data = SummaryData.save(info, SummaryData.Value(1.23, 2.34, 3.45, 4.56))

            # SummaryData のプロパティ
            # Check SummaryData properties
            assert data is not None
            assert data.measure == "dummy"
            assert data.model == "A342"
            assert data.serial == "00000100"
            assert data.physical == "Velocity"
            assert data.value.x == 1.23
            assert data.value.y == 2.34
            assert data.value.z == 3.45
            assert data.value.c == 4.56

            # ファイルを直接確認
            # Verify saved file directly
            expect_path = Path(tmpdir, self._SUMMARY_CSV)
            assert data.path == expect_path

            with open(expect_path, newline="") as f:
                reader = csv.reader(f, quoting=csv.QUOTE_NONNUMERIC)
                lines = list(reader)

            assert lines[0] == [
                "measure",
                "model",
                "serial",
                "physical",
                "X",
                "Y",
                "Z",
                "C",
            ]
            assert lines[1] == [
                "dummy",
                "A342",
                "00000100",
                "Velocity",
                1.23,
                2.34,
                3.45,
                4.56,
            ]

        def test_save_and_load(self, tmpdir):
            info = self._copy_info(tmpdir)

            # save()
            # Save summary data
            data1 = SummaryData.save(info, SummaryData.Value(0.12, 1.23, 2.34, 3.45))

            # load()
            # Load summary data
            data2 = SummaryData.load(info)

            assert data1 is not None
            assert data2 is not None
            assert data1._to_dict() == data2._to_dict()

            # ちゃんと取れているのか
            # Verify correct data retrieval
            assert data1._to_dict() == {
                "measure": "dummy",
                "model": "A342",
                "serial": "00000100",
                "physical": "Velocity",
                "X": 0.12,
                "Y": 1.23,
                "Z": 2.34,
                "C": 3.45,
            }

    class TestDataSet:
        def test_initial_add(self):
            key = SensorKey("A342", "00000100")
            data = TestSummary._create_dummy_summary()

            dataset = DataSet()
            dataset.add(key, data)

            # 検査
            # Verify dataset contents
            count = 0
            for k, v in dataset.iterate():
                count += 1
                assert k == key
                assert len(v) == 1
                d = v[0]._to_dict()
                assert d == {
                    "measure": "dummy",
                    "model": "A342",
                    "serial": "00000100",
                    "physical": "Velocity",
                    "X": 1.23,
                    "Y": 2.34,
                    "Z": 3.45,
                    "C": 4.56,
                }

            assert count == 1

        def test_add_two_same(self):
            key = SensorKey("A342", "00000100")
            data = TestSummary._create_dummy_summary()

            dataset = DataSet()
            dataset.add(key, data)

            # 2 件めを登録
            # Add second entry with same physical type
            data2 = TestSummary._create_dummy_summary()
            data2.value = SummaryData.Value(2.34, 3.45, 4.56, 5.67)
            dataset.add(key, data2)

            # 検査
            # Verify both entries are stored
            count = 0
            for k, v in dataset.iterate():
                count += 1
                assert k == key
                assert len(v) == 2
                assert v[0]._to_dict() == {
                    "measure": "dummy",
                    "model": "A342",
                    "serial": "00000100",
                    "physical": "Velocity",
                    "X": 1.23,
                    "Y": 2.34,
                    "Z": 3.45,
                    "C": 4.56,
                }
                assert v[1]._to_dict() == {
                    "measure": "dummy",
                    "model": "A342",
                    "serial": "00000100",
                    "physical": "Velocity",
                    "X": 2.34,
                    "Y": 3.45,
                    "Z": 4.56,
                    "C": 5.67,
                }

            assert count == 1

        def test_add_two_diff_physical(self, caplog):
            # WARNING 以上のログを取得
            # Capture logs with level WARNING or higher
            caplog.set_level(logging.WARNING)

            key = SensorKey("A342", "00000100")
            data = TestSummary._create_dummy_summary()

            dataset = DataSet()
            dataset.add(key, data)

            # 2 件めを登録
            # Add second entry with different physical type
            data2 = TestSummary._create_dummy_summary()
            data2.value = SummaryData.Value(2.34, 3.45, 4.56, 5.67)
            # 物理量を変更
            # Change physical type
            data2.physical = "Acceleration"

            # - 例外は発生しない
            # - No exception should be raised
            dataset.add(key, data2)

            # ログが出力されていること
            # Verify warning log was emitted
            assert len(caplog.records) == 1
            assert caplog.records[0].levelname == "WARNING"
            assert caplog.records[0].message == (
                "Summary data cannot be used "
                + "because the physical differs from the first: "
                + f"sensor={data.measure}/{key}, "
                + f"first={data.physical}, new={data2.physical}"
            )

            # 検査
            # Verify only the first entry is stored
            count = 0
            for k, v in dataset.iterate():
                count += 1
                assert k == key
                assert len(v) == 1
                assert v[0]._to_dict() == {
                    "measure": "dummy",
                    "model": "A342",
                    "serial": "00000100",
                    "physical": "Velocity",
                    "X": 1.23,
                    "Y": 2.34,
                    "Z": 3.45,
                    "C": 4.56,
                }

            assert count == 1

    class TestSensorSummary:
        def test_create(self):
            sum = SensorSummary()
            assert sum is not None

            # - dataclass(init=False) にしているため属性はまだ存在しない
            # - Attributes are not initialized due to dataclass(init=False)
            # assert sum.datetime is not None
            # assert isinstance(sum.datetime, list)

        def test_to_dict(self):
            sum = SensorSummary()
            sum.datetime = ["20250801_091011", "20250801_101112"]
            sum.physical = "Velocity"
            sum.trend_x = SensorSummary.Axis(
                value=[1.0, 2.0], base=None, limit_alrm=1.5, limit_trip=None
            )
            sum.trend_y = SensorSummary.Axis(
                value=[2.0, 3.0], base=1.0, limit_alrm=None, limit_trip=10.0
            )
            sum.trend_z = SensorSummary.Axis(
                value=[3.0, 4.0], base=2.0, limit_alrm=7.0, limit_trip=11.0
            )
            sum.trend_c = SensorSummary.Axis(
                value=[2.5, 3.5], base=1.5, limit_alrm=6.5, limit_trip=9.5
            )
            sum.change_x = SensorSummary.Axis(
                value=[0.0, 1.0], base=None, limit_alrm=3.0, limit_trip=None
            )
            sum.change_y = SensorSummary.Axis(
                value=[0.0, 1.0], base=None, limit_alrm=None, limit_trip=8.0
            )
            sum.change_z = SensorSummary.Axis(
                value=[0.0, 1.0], base=None, limit_alrm=4.0, limit_trip=9.0
            )
            sum.change_c = SensorSummary.Axis(
                value=[0.0, 1.0], base=None, limit_alrm=5.0, limit_trip=9.5
            )
            sum.alert = "Alarm"

            # to_dict() の出力を検証
            # Verify output of to_dict()
            assert sum.to_dict() == {
                "datetime": ["20250801_091011", "20250801_101112"],
                "physical": "Velocity",
                "trend": {
                    "X": {
                        "value": [1.0, 2.0],
                        "baseline": None,
                        "limit_alarm": 1.5,
                        "limit_trip": None,
                    },
                    "Y": {
                        "value": [2.0, 3.0],
                        "baseline": 1.0,
                        "limit_alarm": None,
                        "limit_trip": 10.0,
                    },
                    "Z": {
                        "value": [3.0, 4.0],
                        "baseline": 2.0,
                        "limit_alarm": 7.0,
                        "limit_trip": 11.0,
                    },
                    "C": {
                        "value": [2.5, 3.5],
                        "baseline": 1.5,
                        "limit_alarm": 6.5,
                        "limit_trip": 9.5,
                    },
                },
                "change": {
                    "X": {
                        "value": [0.0, 1.0],
                        "limit_alarm": 3.0,
                        "limit_trip": None,
                    },
                    "Y": {
                        "value": [0.0, 1.0],
                        "limit_alarm": None,
                        "limit_trip": 8.0,
                    },
                    "Z": {
                        "value": [0.0, 1.0],
                        "limit_alarm": 4.0,
                        "limit_trip": 9.0,
                    },
                    "C": {
                        "value": [0.0, 1.0],
                        "limit_alarm": 5.0,
                        "limit_trip": 9.5,
                    },
                },
                "alert": "Alarm",
            }

    class TestSensorAlert:
        def test_to_dict(self):
            summary = SensorSummary()
            summary.datetime = ["20250801_091011", "20250801_101112"]
            summary.physical = "Velocity"

            alert = SensorAlert(summary)
            alert.trend.append(
                {
                    "axis": "X",
                    "type": "Alarm",
                    "value": 1.234,
                    "baseline": None,
                    "limit_alarm": 0.9,
                    "limit_trip": 2.0,
                }
            )
            alert.trend.append(
                {
                    "axis": "Z",
                    "type": "Trip",
                    "value": 3.456,
                    "baseline": 1.01,
                    "limit_alarm": 2.01,
                    "limit_trip": 3.01,
                }
            )
            alert.level = "Trip"

            assert alert.to_dict() == {
                "datetime": "20250801_101112",
                "physical": "Velocity",
                "level": "Trip",
                "trend": [
                    {
                        "axis": "X",
                        "type": "Alarm",
                        "value": 1.234,
                        "baseline": None,
                        "limit_alarm": 0.9,
                        "limit_trip": 2.0,
                    },
                    {
                        "axis": "Z",
                        "type": "Trip",
                        "value": 3.456,
                        "baseline": 1.01,
                        "limit_alarm": 2.01,
                        "limit_trip": 3.01,
                    },
                ],
                "change": [],
            }

    class TestSensorConfig:
        def test_create(self):
            # 未指定でインスタンス化可能
            # Can be instantiated without arguments
            conf1 = SensorConfig()
            assert conf1 is not None

            # プロパティは保持している
            # Should contain axis properties
            assert conf1.x is not None

            # 値はない
            # Values should be None by default
            assert conf1.x.trend_base is None

    class TestSummaryCalc:
        ZCONF = SummaryConfig(0, 0)

        def test_rms(self):
            lst = [1.0, 2.0, 3.0, 4.0, 5.0]
            # 手計算
            # Manual calculation
            exp = math.sqrt(sum([1.0, 4.0, 9.0, 16.0, 25.0]) / 5)
            assert SummaryCalc.rms(lst) == exp

        def test_composite(self):
            # 三軸合成の計算
            # Composite calculation of three axes
            exp = math.sqrt(1.0**2 + 2.0**2 + 3.0**2)
            assert SummaryCalc.composite(1.0, 2.0, 3.0) == exp

        def _create_dummy_data(self, tmp_path: Path, lines: int = 10) -> Path:
            # ダミーデータを生成
            # Generate dummy CSV data
            bas = [0, 1, 0.0, 0.1, 0.2, 0.4, 6]  # idx, cnt, tmp, x, y, z, flg
            path = Path(tmp_path, "dummy.csv")
            with open(path, mode="w", newline="\r\n") as f:
                writer = csv.writer(f, lineterminator="\n")
                for _ in range(lines):
                    row = bas.copy()
                    writer.writerow(row)

            return path

        def test_summarize_measure_with_insufficient_data(self, tmp_path: Path):
            # ダミー CSV 作成
            # Create dummy CSV with insufficient rows
            path = self._create_dummy_data(tmp_path, lines=99)

            # ダミー INFO 作成
            # Create dummy MeasureInfo
            info = MeasureInfo(Path(tmp_path, "A342_00000100_info.csv"), "dummy")
            info._prop[info._KEY_SPS] = "1"

            # 検査
            # Verify that summary
            cnf = SummaryConfig(0, 100)
            res = SummaryCalc(cnf).summarize_measure(info, MeasureData(path))

            # - 行数が足りないので計算されない
            # - is not calculated due to insufficient data
            assert res is None

        def test_summarize_measure_with_enough_data(self, tmp_path: Path):
            # ダミー CSV 作成
            # Create dummy CSV with enough rows
            path = self._create_dummy_data(tmp_path, lines=100)

            # ダミー INFO 作成
            # Create dummy MeasureInfo with lower SPS
            info = MeasureInfo(Path(tmp_path, "A342_00000100_info.csv"), "dummy")
            info._prop[info._KEY_SPS] = "1"

            # 検査
            # Verify summary calculation
            cnf = SummaryConfig(0, 100)
            res = SummaryCalc(cnf).summarize_measure(info, MeasureData(path))
            exp = SummaryData.Value(0.1, 0.2, 0.4, math.sqrt(0.1**2 + 0.2**2 + 0.4**2))

            assert res is not None

            # - float 計算の加減によってはブレる可能性あり
            # - Allow for small floating-point differences
            def nearly_equal(a: float, b: float) -> bool:
                return round(a, 3) == round(b, 3)

            assert nearly_equal(res.x, exp.x)
            assert nearly_equal(res.y, exp.y)
            assert nearly_equal(res.z, exp.z)
            assert nearly_equal(res.c, exp.c)

        def test_summarize_measure_with_skip_not_enough(self, tmp_path: Path):
            # ダミー CSV 作成
            # Create dummy CSV with insufficient rows
            path = self._create_dummy_data(tmp_path, lines=100)

            # ダミー INFO 作成
            # Create dummy MeasureInfo
            info = MeasureInfo(Path(tmp_path, "A342_00000100_info.csv"), "dummy")
            info._prop[info._KEY_SPS] = "1"

            # 検査
            # Verify that summary
            cnf = SummaryConfig(1, 100)
            res = SummaryCalc(cnf).summarize_measure(info, MeasureData(path))

            # - 行数が足りないので計算されない
            # - is not calculated due to insufficient data
            assert res is None

        def test_summarize_measure_with_skip_enough(self, tmp_path: Path):
            # ダミー CSV 作成
            # Create dummy CSV with insufficient rows
            path = self._create_dummy_data(tmp_path, lines=200)

            # ダミー INFO 作成
            # Create dummy MeasureInfo
            info = MeasureInfo(Path(tmp_path, "A342_00000100_info.csv"), "dummy")
            info._prop[info._KEY_SPS] = "1"

            # 検査
            # Verify that summary
            cnf = SummaryConfig(10, 100)
            res = SummaryCalc(cnf).summarize_measure(info, MeasureData(path))

            # - 行数が足りて計算される
            # - calculated due to enough data
            assert res is not None

        def test_summarize_sensor(self):
            # 1件目
            # First summary data
            data1 = TestSummary._create_dummy_summary()
            data1.measure = "20250915_100000"
            data1.value = SummaryData.Value(1.0, 2.0, 3.0, 4.0)

            # 2件目
            # Second summary data
            data2 = TestSummary._create_dummy_summary()
            data2.measure = "20250916_100000"
            data2.value = SummaryData.Value(1.0 * 2, 2.0 * 2, 3.0 * 2, 4.0 * 2)

            # SensorConfig
            # Configuration for thresholds
            conf = SensorConfig()
            conf.x = SensorConfig.Axis(trend_base=1.0)
            conf.y = SensorConfig.Axis(trend_limit_alrm=3.0)
            conf.z = SensorConfig.Axis(change_limit_alrm=5.0)

            summ, alrt = SummaryCalc(self.ZCONF).summarize_sensor([data1, data2], conf)

            # 抜き打ちで検査
            # Spot check summary values
            assert summ.datetime == ["20250915_100000", "20250916_100000"]
            assert summ.trend_x.value == [1.0, 2.0]
            assert summ.trend_x.base == 1.0
            assert summ.trend_x.limit_alrm is None
            assert summ.trend_y.limit_alrm == 3.0
            assert summ.change_x.value == [0.0, 1.0]
            assert summ.change_y.value == [0.0, 2.0]
            assert summ.change_z.value == [0.0, 3.0]
            assert summ.change_c.value == [0.0, 4.0]
            assert summ.change_z.limit_alrm == 5.0

            # アラートも検査
            # Verify alert contents
            assert alrt.datetime == summ.datetime[-1]
            assert alrt.physical == summ.physical
            assert len(alrt.trend) == 1
            assert alrt.trend[0]["axis"] == "Y"
            assert alrt.trend[0]["type"] == "Alarm"
            assert alrt.trend[0]["value"] == 4.0
            assert alrt.trend[0]["baseline"] is None
            assert alrt.trend[0]["limit_alarm"] == 3.0
            assert alrt.trend[0]["limit_trip"] is None
            assert len(alrt.change) == 0

            # 総合アラートレベル
            # Overall alert level
            assert summ.alert == "Alarm"

        def test_alert_summary(self):
            # ダミーの SensorSummary を構築
            # Construct a dummy SensorSummary
            summary = SensorSummary()
            summary.datetime = ["20250915_100000", "20250916_100000"]
            summary.physical = "Acceleration"
            summary.trend_x = SensorSummary.Axis(
                value=[1.0, 2.0],
                base=1.0,
                limit_alrm=2.0,
                limit_trip=3.0,
            )
            summary.trend_y = SensorSummary.Axis(
                value=[2.0, 4.0],
                base=None,
                limit_alrm=3.0,
                limit_trip=3.5,
            )
            summary.trend_z = SensorSummary.Axis(
                value=[3.0, 6.0],
                base=None,
                limit_alrm=None,
                limit_trip=None,
            )
            summary.trend_c = SensorSummary.Axis(
                value=[4.0, 8.0],
                base=None,
                limit_alrm=None,
                limit_trip=None,
            )
            summary.change_x = SensorSummary.Axis(
                value=[0.0, 1.0],
                base=None,
                limit_alrm=2.0,
                limit_trip=3.0,
            )
            summary.change_y = SensorSummary.Axis(
                value=[0.0, 2.0],
                base=None,
                limit_alrm=2.0,
                limit_trip=3.0,
            )
            summary.change_z = SensorSummary.Axis(
                value=[0.0, 3.0],
                base=None,
                limit_alrm=2.0,
                limit_trip=3.0,
            )
            summary.change_c = SensorSummary.Axis(
                value=[0.0, 4.0],
                base=None,
                limit_alrm=2.0,
                limit_trip=3.0,
            )

            # Alert 判定
            # Perform alert evaluation
            alert = SummaryCalc(self.ZCONF)._alert_summary(summary)

            # 判定
            # Validate alert contents
            assert alert is not None
            assert alert.datetime == "20250916_100000"
            assert alert.physical == "Acceleration"

            assert len(alert.trend) == 2
            t0 = alert.trend[0]
            assert t0["axis"] == "X"
            assert t0["type"] == "Alarm"
            assert t0["baseline"] == 1.0
            assert t0["limit_alarm"] == 2.0
            assert t0["limit_trip"] == 3.0
            t1 = alert.trend[1]
            assert t1["axis"] == "Y"
            assert t1["type"] == "Trip"
            assert t1["baseline"] is None
            assert t1["limit_trip"] == 3.5

            assert len(alert.change) == 3
            assert alert.change[0]["axis"] == "Y"
            assert alert.change[0]["type"] == "Alarm"
            assert "baseline" not in alert.change[0]
            assert alert.change[0]["limit_alarm"] == 2.0
            assert alert.change[0]["limit_trip"] == 3.0
            assert alert.change[1]["axis"] == "Z"
            assert alert.change[1]["type"] == "Trip"
            assert alert.change[2]["axis"] == "C"
            assert alert.change[2]["type"] == "Trip"

            assert alert.level == "Trip"

            assert summary.alert == "Trip"

        def test_alert_summary_half_none(self):
            # ダミーの SensorSummary を構築
            # Construct a dummy SensorSummary with partial thresholds
            summary = SensorSummary()
            summary.datetime = ["20250915_100000", "20250916_100000"]
            summary.physical = "Velocity"
            summary.trend_x = SensorSummary.Axis(
                value=[5.0, 2.0],
                base=1.0,
                limit_alrm=2.0,
                limit_trip=3.0,
            )
            summary.trend_y = SensorSummary.Axis(
                value=[2.0, 4.0],
                base=None,
                limit_alrm=None,
                limit_trip=None,
            )
            summary.trend_z = SensorSummary.Axis(
                value=[3.0, 6.0],
                base=None,
                limit_alrm=None,
                limit_trip=None,
            )
            summary.trend_c = SensorSummary.Axis(
                value=[4.0, 8.0],
                base=None,
                limit_alrm=None,
                limit_trip=None,
            )
            summary.change_x = SensorSummary.Axis(
                value=[0.0, 1.0],
                base=None,
                limit_alrm=None,
                limit_trip=None,
            )
            summary.change_y = SensorSummary.Axis(
                value=[0.0, 2.0],
                base=None,
                limit_alrm=None,
                limit_trip=None,
            )
            summary.change_z = SensorSummary.Axis(
                value=[0.0, 3.0],
                base=None,
                limit_alrm=None,
                limit_trip=None,
            )
            summary.change_c = SensorSummary.Axis(
                value=[0.0, 4.0],
                base=None,
                limit_alrm=None,
                limit_trip=None,
            )

            # Alert 判定
            # Perform alert evaluation
            alert = SummaryCalc(self.ZCONF)._alert_summary(summary)

            # 判定
            # Validate alert contents
            assert alert is not None
            assert alert.datetime == "20250916_100000"
            assert alert.physical == "Velocity"

            assert len(alert.trend) == 1
            assert alert.trend[0]["axis"] == "X"
            assert alert.trend[0]["type"] == "Alarm"

            assert len(alert.change) == 0

            assert alert.level == "Alarm"

            # Summary は全体で判定
            # Summary is validated as a whole
            assert summary.alert == "Trip"

        def test_alert_summary_all_none(self):
            # ダミーの SensorSummary を構築
            # Construct a dummy SensorSummary with no thresholds
            summary = SensorSummary()
            summary.datetime = ["20250915_100000", "20250916_100000"]
            summary.physical = "Velocity"
            summary.trend_x = SensorSummary.Axis(
                value=[1.0, 1.9],
                base=1.0,
                limit_alrm=2.0,
                limit_trip=3.0,
            )
            summary.trend_y = SensorSummary.Axis(
                value=[2.0, 4.0],
                base=None,
                limit_alrm=None,
                limit_trip=None,
            )
            summary.trend_z = SensorSummary.Axis(
                value=[3.0, 6.0],
                base=None,
                limit_alrm=None,
                limit_trip=None,
            )
            summary.trend_c = SensorSummary.Axis(
                value=[4.0, 8.0],
                base=None,
                limit_alrm=None,
                limit_trip=None,
            )
            summary.change_x = SensorSummary.Axis(
                value=[0.0, 1.0],
                base=None,
                limit_alrm=None,
                limit_trip=None,
            )
            summary.change_y = SensorSummary.Axis(
                value=[0.0, 2.0],
                base=None,
                limit_alrm=None,
                limit_trip=None,
            )
            summary.change_z = SensorSummary.Axis(
                value=[0.0, 3.0],
                base=None,
                limit_alrm=None,
                limit_trip=None,
            )
            summary.change_c = SensorSummary.Axis(
                value=[0.0, 4.0],
                base=None,
                limit_alrm=None,
                limit_trip=None,
            )

            # Alert 判定
            # Perform alert evaluation
            alert = SummaryCalc(self.ZCONF)._alert_summary(summary)

            # 判定
            # Validate alert contents
            assert alert is not None
            assert alert.trend == []
            assert alert.change == []
            assert alert.level == ""

            # Summary もなし
            # Summary also no alert
            assert summary.alert == ""

        def test_trend_to_change(self):
            tre = [float(v) for v in range(5)]
            chg = SummaryCalc.trend_to_change(tre)

            assert chg == [0.0, 1.0, 1.0, 1.0, 1.0]

        def test_trend_to_change_with_minus(self):
            tre = [1.0, 5.0, 3.0, 1.0, 0.0]
            chg = SummaryCalc.trend_to_change(tre)

            assert chg == [0.0, 4.0, 2.0, 2.0, 1.0]

        def test_over_limit(self):
            val = [float(v) for v in range(5)]
            res = SummaryCalc.over_limit(val, 3.0)

            assert res is not None
            assert res == [False, False, False, False, True]

        def test_over_limit_with_none(self):
            val = [float(v) for v in range(5)]
            res = SummaryCalc.over_limit(val, None)

            assert res is None

    class TestAlertScanner:

        def test_slice_latest(self):
            # 検査対象
            # Test target
            lst = [0, 1, 2, 3, 4]

            # 最後の1個を取得する slice で取得
            # Use a slice to get the last item
            cut = lst[slice(-1, -2, -1)]

            # 1個のみ取れていること
            # Ensure only one item is retrieved
            assert len(cut) == 1

            # 最後が取れていること
            # Ensure the last item is retrieved
            assert cut[0] == 4

        def test_max_alert(self):
            scanner = _AlertScanner(SensorSummary())

            # 何もない場合、"" が取得されること
            # Ensure "" is retrieved when empty
            assert scanner._max_alert() == ""

            # AlertType を追加
            # Add AlertTypes
            scanner._alerts.append("Alarm")
            scanner._alerts.append("")

            # Alarm が取得されること
            # Ensure Alarm is retrieved
            assert scanner._max_alert() == "Alarm"

            # Trip を追加
            # Add Trip
            scanner._alerts.append("Trip")

            # Trip が取得されること
            # Ensure Trip is retrieved
            assert scanner._max_alert() == "Trip"
