# Copyright 2025, Seiko Epson Corporation
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the “Software”), to deal in
# the Software without restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
# Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

import logging
import os
from pathlib import Path

import pytest

import raspi_summary.config as config
from raspi_summary.const import Const


def _putenv(key: str, val: str) -> None:
    """
    環境変数に追加する

    Add an environment variable
    """
    os.environ[key] = val


@pytest.mark.usefixtures("clear_env")
class TestConfig:
    class TestLoggerId:
        def test_not_defined(self, caplog):
            # WARNING 以上のログを取得
            # Capture logs with level WARNING or higher
            caplog.set_level(logging.WARNING)

            # ないこと
            # Ensure LOGGER_ID is not set
            assert os.getenv("LOGGER_ID") is None

            # 実行
            # Execute logger_id()
            val = config.logger_id()

            # 値が取得されない
            # Value should not be retrieved
            assert val is None

            # ログが出力されていること
            # Verify error log is emitted
            assert len(caplog.records) == 1
            assert caplog.records[0].levelname == "ERROR"
            assert caplog.records[0].message == "LOGGER_ID must be defined"

        def test_defined(self, caplog):
            caplog.set_level(logging.WARNING)

            # 値をセット
            # Set LOGGER_ID
            _putenv("LOGGER_ID", "RP1")

            # 実行
            # Execute logger_id()
            val = config.logger_id()

            # 値が取得される
            # Value should be retrieved
            assert val == "RP1"

            # ログが出力されていないこと
            # No log should be emitted
            assert len(caplog.records) == 0

    class TestOutputPath:
        def test_not_defined(self, caplog):
            # WARNING 以上のログを取得
            # Capture logs with level WARNING or higher
            caplog.set_level(logging.WARNING)

            # ないこと
            # Ensure OUTPUT_PATH is not set
            assert os.getenv("OUTPUT_PATH") is None

            # 実行
            # Execute output_path()
            val = config.output_path()

            # 値が取得されない
            # Value should not be retrieved
            assert val is None

            # ログが出力されていること
            # Verify error log is emitted
            assert len(caplog.records) == 1
            assert caplog.records[0].levelname == "ERROR"
            assert caplog.records[0].message == "OUTPUT_PATH must be defined"

        def test_defined_exists(self, caplog):
            caplog.set_level(logging.WARNING)

            # 存在するディレクトリをセット
            # Set an existing directory
            path = str(Path("./tests").resolve())
            assert path.startswith("/")
            assert os.path.exists(path)
            _putenv("OUTPUT_PATH", path)

            # 実行
            # Execute output_path()
            val = config.output_path()

            # 値が取得される
            # Value should be retrieved
            assert val == path

            # ログが出力されていないこと
            # No log should be emitted
            assert len(caplog.records) == 0

        def test_defined_not_exist(self, caplog):
            caplog.set_level(logging.WARNING)

            # 存在しないディレクトリをセット
            # Set a non-existent directory
            path = str(Path("./no_such_dir").resolve())
            assert path.startswith("/")
            assert not os.path.exists(path)
            _putenv("OUTPUT_PATH", path)

            # 実行
            # Execute output_path()
            val = config.output_path()

            # 値が取得されない
            # Value should not be retrieved
            assert val is None

            # ログが出力されていること
            # Verify error log is emitted
            assert len(caplog.records) == 1
            assert caplog.records[0].levelname == "ERROR"
            assert caplog.records[0].message.startswith(
                "OUTPUT_PATH must be an existing directory: /"
            )

    class TestMaxMeasurements:
        def test_not_defined(self, caplog):
            caplog.set_level(logging.WARNING)

            # ないこと
            # Ensure MAX_MEASUREMENTS is not set
            assert os.getenv("MAX_MEASUREMENTS") is None

            # 実行
            # Execute max_measurements()
            val = config.max_measurements()

            # 値が取得されない
            # Value should not be retrieved
            assert val is None

            # ログが出力されていること
            # Verify error log is emitted
            assert len(caplog.records) == 1
            assert caplog.records[0].levelname == "ERROR"
            assert caplog.records[0].message == "MAX_MEASUREMENTS must be defined"

        def test_defined(self, caplog):
            caplog.set_level(logging.WARNING)

            # 値をセット
            # Set valid value
            _putenv("MAX_MEASUREMENTS", "10")

            # 実行
            # Execute max_measurements()
            val = config.max_measurements()

            # 値が取得される
            # Value should be retrieved
            assert val is not None
            assert val == 10

            # ログが出力されていないこと
            # No log should be emitted
            assert len(caplog.records) == 0

        def test_defined_less(self, caplog):
            caplog.set_level(logging.WARNING)

            # 値をセット
            # Set valid value
            _putenv("MAX_MEASUREMENTS", "-1")

            # 実行
            # Execute max_measurements()
            val = config.max_measurements()

            # 値が取得される
            # Value should not be retrieved
            assert val is None

            # ログが出力されていること
            # Verify error log is emitted
            msg = "{} must be greater than or equal to {}"
            assert len(caplog.records) == 1
            assert caplog.records[0].levelname == "ERROR"
            assert caplog.records[0].message == msg.format("MAX_MEASUREMENTS", 0)

        def test_defined_invalid(self, caplog):
            caplog.set_level(logging.WARNING)

            # 値をセット
            # Set valid value
            _putenv("MAX_MEASUREMENTS", "hoge")

            # 実行
            # Execute max_measurements()
            val = config.max_measurements()

            # 値が取得される
            # Value should not be retrieved
            assert val is None

            # ログが出力されていること
            # Verify error log is emitted
            assert len(caplog.records) == 1
            assert caplog.records[0].levelname == "ERROR"
            assert caplog.records[0].message == "MAX_MEASUREMENTS must be an integer"

    class TestSensorModel:
        def test_not_defined(self, caplog):
            caplog.set_level(logging.WARNING)

            # ないこと
            # Ensure SENSOR_MODEL is not set
            assert os.getenv("SENSOR_MODEL") is None

            # 実行
            # Execute sensor_model()
            val = config.sensor_model()

            # 値が取得されない
            # Value should not be retrieved
            assert val is None

            # ログが出力されていること
            # Verify error log is emitted
            assert len(caplog.records) == 1
            assert caplog.records[0].levelname == "ERROR"
            assert caplog.records[0].message == "SENSOR_MODEL must be defined"

        def test_defined(self, caplog):
            caplog.set_level(logging.WARNING)

            # 値をセット
            # Set SENSOR_MODEL
            _putenv("SENSOR_MODEL", "A342,A352")

            # 実行
            # Execute sensor_model()
            val = config.sensor_model()

            # 値が取得される
            # Value should be retrieved
            assert val == "A342,A352"

            # ログが出力されていないこと
            # No log should be emitted
            assert len(caplog.records) == 0

    class TestMessageConfig:
        def test_not_defined(self, caplog):
            caplog.set_level(logging.WARNING)

            # ないこと
            # Ensure MESSAGE_HOST, MESSAGE_PORT and MESSAGE_CLID are not set
            assert os.getenv("MESSAGE_HOST") is None
            assert os.getenv("MESSAGE_PORT") is None
            assert os.getenv("MESSAGE_CLID") is None

            # 実行
            # Execute message_config()
            conf = config.message_config()

            # 器は取得される
            # Config object should be returned
            assert conf is not None
            # 中身はない
            # Values should be None
            assert conf.host is None
            assert conf.port is None
            assert conf.clid is None

            # ログが出力されていること
            # Verify warning logs are emitted
            assert len(caplog.records) == 2
            assert caplog.records[0].levelname == "WARNING"
            assert caplog.records[0].message == "MESSAGE_HOST is not defined"
            assert caplog.records[1].levelname == "WARNING"
            assert caplog.records[1].message == "MESSAGE_PORT is not defined"

            # MESSAGE_CLID へのログはない
            # No warning for MESSAGE_CLID

        def test_defined_host(self, caplog):
            caplog.set_level(logging.WARNING)

            # ないこと
            # Ensure MESSAGE_HOST and MESSAGE_PORT are not set
            assert os.getenv("MESSAGE_HOST") is None
            assert os.getenv("MESSAGE_PORT") is None

            # 値をセット
            # Set MESSAGE_HOST only
            _putenv("MESSAGE_HOST", "hoge")

            # 実行
            # Execute message_config()
            conf = config.message_config()

            # 値が取得される
            # Host should be retrieved
            assert conf is not None
            assert conf.host == "hoge"
            # ないものはない
            # Port should be None
            assert conf.port is None

            # ログが出力されていること
            # Verify warning log for missing port
            assert len(caplog.records) == 1
            assert caplog.records[0].levelname == "WARNING"
            assert caplog.records[0].message == "MESSAGE_PORT is not defined"

        def test_defined_port(self, caplog):
            caplog.set_level(logging.WARNING)

            # ないこと
            # Ensure MESSAGE_HOST and MESSAGE_PORT are not set
            assert os.getenv("MESSAGE_HOST") is None
            assert os.getenv("MESSAGE_PORT") is None

            # 値をセット
            # Set MESSAGE_PORT only
            _putenv("MESSAGE_PORT", "100")

            # 実行
            # Execute message_config()
            conf = config.message_config()

            # 値が取得される
            # Port should be retrieved
            assert conf is not None
            assert conf.port == 100
            # ないものはない
            # Host should be None
            assert conf.host is None

            # ログが出力されていること
            # Verify warning log for missing host
            assert len(caplog.records) == 1
            assert caplog.records[0].levelname == "WARNING"
            assert caplog.records[0].message == "MESSAGE_HOST is not defined"

        def test_defined_port_less(self, caplog):
            caplog.set_level(logging.WARNING)

            # ないこと
            # Ensure MESSAGE_HOST and MESSAGE_PORT are not set
            assert os.getenv("MESSAGE_HOST") is None
            assert os.getenv("MESSAGE_PORT") is None

            # 値をセット
            # Set invalid port value
            _putenv("MESSAGE_HOST", "localhost")
            _putenv("MESSAGE_PORT", "-1")

            # 実行
            # Execute message_config()
            conf = config.message_config()

            # 値が取得される
            # Host should be retrieved
            assert conf is not None
            assert conf.host == "localhost"
            # 不正は None
            # Invalid port should be None
            assert conf.port is None

            # ログが出力されていること
            # Verify error log for invalid port
            assert len(caplog.records) == 1
            assert caplog.records[0].levelname == "ERROR"
            assert caplog.records[0].message == "MESSAGE_PORT must be greater than 0"

        def test_defined_invalid(self, caplog):
            caplog.set_level(logging.WARNING)

            # ないこと
            # Ensure MESSAGE_HOST and MESSAGE_PORT are not set
            assert os.getenv("MESSAGE_HOST") is None
            assert os.getenv("MESSAGE_PORT") is None

            # 値をセット
            # Set non-numeric port value
            _putenv("MESSAGE_HOST", "localhost")
            _putenv("MESSAGE_PORT", "hoge")

            # 実行
            # Execute message_config()
            conf = config.message_config()

            # 値が取得される
            # Host should be retrieved
            assert conf is not None
            assert conf.host == "localhost"
            # 不正は None
            # Invalid port should be None
            assert conf.port is None

            # ログが出力されていること
            # Verify error log for non-numeric port
            assert len(caplog.records) == 1
            assert caplog.records[0].levelname == "ERROR"
            assert caplog.records[0].message == "MESSAGE_PORT must be an integer"

        def test_defined_clid(self):
            # ないこと
            # Ensure MESSAGE_CLID is not set
            assert os.getenv("MESSAGE_CLID") is None

            # 値をセット
            # Set MESSAGE_CLID
            _putenv("MESSAGE_CLID", "hoge")

            # 実行
            # Execute message_config()
            conf = config.message_config()

            # 値が取得される
            # CLID should be retrieved
            assert conf is not None
            assert conf.clid == "hoge"

    class TestSummaryConfig:
        def test_not_defined(self, caplog):
            caplog.set_level(logging.WARNING)

            # ないこと
            # Ensure variables are not set
            assert os.getenv("SKIP_SECONDS") is None
            assert os.getenv("SUMMARY_SECONDS") is None

            # 実行
            # Execute summary_config()
            conf = config.summary_config()

            # 設定が取得されない
            # Configuration should not be retrieved
            assert conf is None

            # ログが出力されていること
            # Verify error log is emitted
            assert len(caplog.records) == 2
            assert caplog.records[0].levelname == "ERROR"
            assert caplog.records[0].message == "SKIP_SECONDS must be defined"
            assert caplog.records[1].levelname == "ERROR"
            assert caplog.records[1].message == "SUMMARY_SECONDS must be defined"

        def test_defined(self, caplog):
            caplog.set_level(logging.WARNING)

            # 値をセット
            # Set valid value
            _putenv("SKIP_SECONDS", "10")
            _putenv("SUMMARY_SECONDS", "30")

            # 実行
            # Execute summary_config()
            conf = config.summary_config()

            # 値が取得される
            # Value should be retrieved
            assert conf is not None
            assert conf.skip_seconds == 10
            assert conf.summary_seconds == 30

            # ログが出力されていないこと
            # No log should be emitted
            assert len(caplog.records) == 0

        def test_defined_with_zero(self, caplog):
            caplog.set_level(logging.WARNING)

            # 値をセット
            # Set valid value
            _putenv("SKIP_SECONDS", "0")
            _putenv("SUMMARY_SECONDS", "30")

            # 実行
            # Execute summary_config()
            conf = config.summary_config()

            # 値が取得される
            # Value should be retrieved
            assert conf is not None
            assert conf.skip_seconds == 0
            assert conf.summary_seconds == 30

            # ログが出力されていないこと
            # No log should be emitted
            assert len(caplog.records) == 0

        def test_defined_less(self, caplog):
            caplog.set_level(logging.WARNING)

            # 値をセット
            # Set values below valid range
            _putenv("SKIP_SECONDS", "-1")
            _putenv("SUMMARY_SECONDS", "0")

            # 実行
            # Execute summary_config()
            conf = config.summary_config()

            # 設定が取得されない
            # Configuration should not be retrieved
            assert conf is None

            # ログが出力されていること
            # Verify error log is emitted
            msg = "{} must be greater than or equal to {}"
            assert len(caplog.records) == 2
            assert caplog.records[0].levelname == "ERROR"
            assert caplog.records[0].message == msg.format("SKIP_SECONDS", 0)
            assert caplog.records[1].levelname == "ERROR"
            assert caplog.records[1].message == msg.format("SUMMARY_SECONDS", 1)

        def test_defined_invalid(self, caplog):
            caplog.set_level(logging.WARNING)

            # 値をセット
            # Set non-numeric value
            _putenv("SKIP_SECONDS", "hoge")
            _putenv("SUMMARY_SECONDS", "hoge")

            # 実行
            # Execute summary_config()
            conf = config.summary_config()

            # 値が取得されない
            # Value should not be retrieved
            assert conf is None

            # ログが出力されていること
            # Verify error log is emitted
            assert len(caplog.records) == 2
            assert caplog.records[0].levelname == "ERROR"
            assert caplog.records[0].message == "SKIP_SECONDS must be an integer"
            assert caplog.records[1].levelname == "ERROR"
            assert caplog.records[1].message == "SUMMARY_SECONDS must be an integer"

    class TestSensorConfig:
        def test_not_defined(self, caplog):
            caplog.set_level(logging.WARNING)

            # ないこと
            # Ensure all environment variables for sensor monitoring are unset
            key = "A342_00000100"
            for axis in Const.AXIS_VALUES:
                assert os.getenv(f"{key}_{axis}_TREND_BASE") is None
                assert os.getenv(f"{key}_{axis}_TREND_ALARM") is None
                assert os.getenv(f"{key}_{axis}_TREND_TRIP") is None
                assert os.getenv(f"{key}_{axis}_CHANGE_ALARM") is None
                assert os.getenv(f"{key}_{axis}_CHANGE_TRIP") is None

            # 実行
            # Execute sensor_config()
            sconf = config.sensor_config("A342", "00000100")

            # ないこと
            # All values should be None
            for aconf in [sconf.x, sconf.y, sconf.z, sconf.c]:
                assert aconf.trend_base is None
                assert aconf.trend_limit_alrm is None
                assert aconf.trend_limit_trip is None
                assert aconf.change_limit_alrm is None
                assert aconf.change_limit_trip is None

            # ログが出力されていること
            # Verify warning logs are emitted for all missing keys
            assert len(caplog.records) == 4 * 5
            assert caplog.records[0].levelname == "WARNING"
            assert caplog.records[0].message == f"{key}_X_TREND_BASE is not defined"

            # 残り
            # Remaining logs should also be warnings for undefined keys
            for i in range(1, len(caplog.records)):
                assert caplog.records[i].levelname == "WARNING"
                assert caplog.records[i].message.endswith(" is not defined")

        def test_defined_mixed(self, caplog):
            caplog.set_level(logging.WARNING)

            # ないこと
            # Ensure all environment variables for sensor monitoring are unset
            key = "A342_00000100"
            for axis in Const.AXIS_VALUES:
                assert os.getenv(f"{key}_{axis}_TREND_BASE") is None
                assert os.getenv(f"{key}_{axis}_TREND_ALARM") is None
                assert os.getenv(f"{key}_{axis}_TREND_TRIP") is None
                assert os.getenv(f"{key}_{axis}_CHANGE_ALARM") is None
                assert os.getenv(f"{key}_{axis}_CHANGE_TRIP") is None

            # 値をさまざまセット
            # Set various values with mixed validity
            _putenv(f"{key}_X_TREND_BASE", "hoge")  # invalid string
            _putenv(f"{key}_Y_TREND_ALARM", "100")  # valid int
            _putenv(f"{key}_Z_TREND_TRIP", "1.01")  # valid float
            _putenv(f"{key}_C_CHANGE_ALARM", "-1")  # valid negative
            _putenv(f"{key}_C_CHANGE_TRIP", "0.1")  # valid float

            # 実行
            # Execute sensor_config()
            sconf = config.sensor_config("A342", "00000100")

            # 設定したものをチェック
            # Check values that were set
            assert sconf.x.trend_base is None
            assert sconf.y.trend_limit_alrm is not None
            assert sconf.z.trend_limit_trip is not None
            assert sconf.c.change_limit_alrm is not None
            assert sconf.c.change_limit_trip is not None

            # float 化チェック
            # Check float values
            assert sconf.y.trend_limit_alrm == 100.0
            assert sconf.z.trend_limit_trip == 1.01
            assert sconf.c.change_limit_alrm == -1.0
            assert sconf.c.change_limit_trip == 0.1

            # ないものカウント
            # Count how many values are still None
            none_count = 0
            for aconf in [sconf.x, sconf.y, sconf.z, sconf.c]:
                none_count += 1 if aconf.trend_base is None else 0
                none_count += 1 if aconf.trend_limit_alrm is None else 0
                none_count += 1 if aconf.trend_limit_trip is None else 0
                none_count += 1 if aconf.change_limit_alrm is None else 0
                none_count += 1 if aconf.change_limit_trip is None else 0
            assert none_count == 20 - 4

            # ないものにはログ
            # Logs should be emitted for missing or invalid values
            assert len(caplog.records) == 20 - 4

            # - 最初はエラー
            # First log should be an error for invalid numeric value
            assert caplog.records[0].levelname == "ERROR"
            assert caplog.records[0].message == f"{key}_X_TREND_BASE must be numeric"

            # - 他は WARNING
            # Remaining logs should be warnings for undefined values
            for i in range(1, 20 - 4):
                assert caplog.records[i].levelname == "WARNING"
                assert caplog.records[i].message.endswith(" is not defined")
